/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.util.NamingUtil.isFriendlyName;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ParameterDescriptor;
import org.mule.connectivity.restconnect.internal.util.NamingUtil;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

public class ConnectorParameterBuilder {

  private final ConnectorTypeDefinitionBuilder typeDefinitionBuilder;

  public ConnectorParameterBuilder(ConnectorTypeDefinitionBuilder connectorTypeDefinitionBuilder) {
    this.typeDefinitionBuilder = connectorTypeDefinitionBuilder;
  }

  public List<Parameter> buildParameterList(List<APIParameterModel> parametersModel,
                                            OperationDescriptor operationDescriptor,
                                            List<String> parameterCollisionInternalNames)
      throws ModelGenerationException {

    List<Parameter> parameterList = new ArrayList<>();
    List<String> collisionNames = new LinkedList<>(parameterCollisionInternalNames);

    for (APIParameterModel parameterModel : parametersModel) {
      Parameter parameter =
          buildParameter(parameterModel, getParameterDescriptor(parameterModel, operationDescriptor), collisionNames);
      parameterList.add(parameter);
      collisionNames.add(parameter.getInternalName());
    }

    return parameterList;
  }

  private static ParameterDescriptor getParameterDescriptor(APIParameterModel parameterModel,
                                                            OperationDescriptor operationDescriptor) {
    if (operationDescriptor == null) {
      return null;
    }

    List<ParameterDescriptor> parameterDescriptors = new LinkedList<>();

    if (operationDescriptor.getExpects() != null) {
      if (parameterModel.getParameterType().equals(ParameterType.URI)) {
        parameterDescriptors = operationDescriptor.getExpects().getUriParameter();
      } else if (parameterModel.getParameterType().equals(ParameterType.QUERY)) {
        parameterDescriptors = operationDescriptor.getExpects().getQueryParameter();
      } else if (parameterModel.getParameterType().equals(ParameterType.HEADER)) {
        parameterDescriptors = operationDescriptor.getExpects().getHeader();
      }
    }

    return parameterDescriptors.stream()
        .filter(x -> isNotBlank(x.getParamName()))
        .filter(x -> x.getParamName().equalsIgnoreCase(parameterModel.getExternalName()))
        .findFirst().orElse(null);
  }

  public Parameter buildParameter(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor,
                                  List<String> collisionInternalNames)
      throws ModelGenerationException {

    return new Parameter(buildDisplayName(apiParameterModel, parameterDescriptor),
                         apiParameterModel.getExternalName(),
                         apiParameterModel.getParameterType(),
                         buildTypeDefinition(apiParameterModel.getTypeModel()),
                         buildDescription(apiParameterModel, parameterDescriptor),
                         apiParameterModel.isRequired(),
                         apiParameterModel.getDefaultValue(),
                         apiParameterModel.isPassword(),
                         collisionInternalNames);
  }

  private static String buildDisplayName(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor) {
    String displayName;

    if (parameterDescriptor != null && isNotBlank(parameterDescriptor.getParamName())) {
      displayName = parameterDescriptor.getFriendlyName();
    } else {
      displayName = apiParameterModel.getDisplayName();
    }

    if (!isFriendlyName(displayName)) {
      displayName = NamingUtil.makeNameFriendly(displayName);
    }

    return displayName;
  }

  private static String buildDescription(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor) {
    if (parameterDescriptor != null && isNotBlank(parameterDescriptor.getParamName())) {
      return parameterDescriptor.getDescription();
    } else {
      return apiParameterModel.getDescription();
    }
  }

  private TypeDefinition buildTypeDefinition(APITypeModel typeModel)
      throws ModelGenerationException {
    return typeDefinitionBuilder.buildTypeDefinition(typeModel);
  }

  public PartParameter buildPartParameter(APIParameterModel apiParameterModel) throws ModelGenerationException {
    return new PartParameter(
                             apiParameterModel.getExternalName(),
                             apiParameterModel.getParameterType(),
                             typeDefinitionBuilder.buildTypeDefinition(apiParameterModel.getTypeModel()));
    //TODO: RESTC-665: Handle multipart requests
  }
}
