/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static java.util.stream.Collectors.toList;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.BasicAuthScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.CustomAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.DigestAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.NotSupportedScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2AuthorizationCodeScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2ClientCredentialsScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.PassThroughScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.UnsecuredScheme;
import org.mule.connectivity.restconnect.internal.webapi.model.APISecuritySchemeModel;

import java.util.LinkedList;
import java.util.List;

public class ConnectorSecuritySchemeBuilder {

  private final ConnectorParameterBuilder parameterBuilder;

  public ConnectorSecuritySchemeBuilder(ConnectorTypeDefinitionBuilder connectorTypeDefinitionBuilder) {
    this.parameterBuilder = new ConnectorParameterBuilder(connectorTypeDefinitionBuilder);
  }

  public List<APISecurityScheme> buildSecuritySchemes(List<APISecuritySchemeModel> apiSecuritySchemeModels)
      throws ModelGenerationException {
    List<APISecurityScheme> apiSecuritySchemes = new LinkedList<>();

    if (!apiSecuritySchemeModels.isEmpty()) {
      for (APISecuritySchemeModel securitySchemeModel : apiSecuritySchemeModels) {
        APISecurityScheme apiSecurityScheme = buildSecurityScheme(securitySchemeModel);
        if (!(apiSecurityScheme instanceof NotSupportedScheme)) {
          apiSecuritySchemes.add(apiSecurityScheme);
        }
      }
    } else {
      apiSecuritySchemes.add(new UnsecuredScheme());
    }

    return apiSecuritySchemes;
  }

  private APISecurityScheme buildSecurityScheme(APISecuritySchemeModel securitySchemeModel)
      throws ModelGenerationException {
    Class<?> securitySchemeClass = securitySchemeModel.getSecuritySchemeClass();

    List<String> collisionNames = new LinkedList<>();
    List<Parameter> customQueryParams =
        parameterBuilder.buildParameterList(securitySchemeModel.getCustomQueryParams(), null, collisionNames);
    collisionNames.addAll(customQueryParams.stream().map(Parameter::getInternalName).collect(toList()));
    List<Parameter> customHeaders =
        parameterBuilder.buildParameterList(securitySchemeModel.getCustomHeaders(), null, collisionNames);

    APISecurityScheme.Builder builder =
        APISecurityScheme.builder(securitySchemeModel.getDisplayName(), customQueryParams, customHeaders);

    if (securitySchemeClass.equals(BasicAuthScheme.class)) {
      return builder.buildBasicAuthScheme();
    } else if (securitySchemeClass.equals(CustomAuthenticationScheme.class)) {
      return builder.buildCustomAuthenticationScheme();
    } else if (securitySchemeClass.equals(DigestAuthenticationScheme.class)) {
      return builder.buildDigestAuthenticationSchemeScheme();
    } else if (securitySchemeClass.equals(OAuth2AuthorizationCodeScheme.class)) {
      return builder.buildOAuth2AuthorizationCodeScheme(securitySchemeModel.getAuthorizationUri(),
                                                        securitySchemeModel.getAccessTokenUri(),
                                                        securitySchemeModel.getScopes());
    } else if (securitySchemeClass.equals(OAuth2ClientCredentialsScheme.class)) {
      return builder.buildOAuth2ClientCredentialsScheme(securitySchemeModel.getAuthorizationUri(),
                                                        securitySchemeModel.getAccessTokenUri(),
                                                        securitySchemeModel.getScopes());
    } else if (securitySchemeClass.equals(PassThroughScheme.class)) {
      return builder.buildPassThroughScheme();
    } else {
      return builder.buildUnsecuredScheme();
    }
  }
}
