/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.isXmlSchema;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.type.ArrayTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.EmptyTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.MultipartTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.ObjectTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.UnionTypeDefinition;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIPrimitiveTypeModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.TypeSchemaPool;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.core.MediaType;

public class ConnectorTypeDefinitionBuilder {

  private final ConnectorTypeSchemaBuilder typeSchemaBuilder;
  private final ConnectorParameterBuilder parameterBuilder;

  public ConnectorTypeDefinitionBuilder(TypeSchemaPool typeSchemaPool) {
    this.typeSchemaBuilder = new ConnectorTypeSchemaBuilder(typeSchemaPool);
    this.parameterBuilder = new ConnectorParameterBuilder(this);
  }

  public TypeDefinition buildTypeDefinition(APITypeModel apiTypeModel) throws ModelGenerationException {
    TypeDefinition.Builder typeDefinitionBuilder =
        TypeDefinition.builder(apiTypeModel.getMediaType(),
                               typeSchemaBuilder.buildTypeSchema(apiTypeModel.getAPITypeSchemaModel()),
                               apiTypeModel.getEnumValues(),
                               apiTypeModel.getExample());

    Class<?> typeDefinitionClass = apiTypeModel.getTypeDefinitionClass();

    if (typeDefinitionClass == null) {
      return null;
    } else if (typeDefinitionClass.equals(EmptyTypeDefinition.class)) {
      return typeDefinitionBuilder.buildEmptyTypeDefinition();
    } else if (typeDefinitionClass.equals(PrimitiveTypeDefinition.class)) {
      return typeDefinitionBuilder.buildPrimitiveTypeDefinition(buildPrimitiveType(apiTypeModel.getPrimitiveTypeModel()));
    } else if (typeDefinitionClass.equals(ObjectTypeDefinition.class)) {
      return typeDefinitionBuilder.buildObjectTypeDefinition();
    } else if (typeDefinitionClass.equals(ArrayTypeDefinition.class)) {
      return typeDefinitionBuilder.buildArrayType(buildTypeDefinition(apiTypeModel.getInnerType()));
    } else if (typeDefinitionClass.equals(UnionTypeDefinition.class)) {
      List<TypeDefinition> unionTypes = new ArrayList<>();
      for (APITypeModel typeModel : apiTypeModel.getUnionTypes()) {
        TypeDefinition typeDefinition = buildTypeDefinition(typeModel);
        unionTypes.add(typeDefinition);
      }
      return typeDefinitionBuilder.buildUnionTypeDefinition(unionTypes);
    } else if (typeDefinitionClass.equals(MultipartTypeDefinition.class)) {
      List<PartParameter> parts = new ArrayList<>();
      for (APIParameterModel apiParameterModel : apiTypeModel.getParts()) {
        PartParameter partParameter = parameterBuilder.buildPartParameter(apiParameterModel);
        parts.add(partParameter);
      }
      return typeDefinitionBuilder.buildMultipartTypeDefinition(parts);
    } else {
      return null;
    }
  }

  public TypeDefinition buildTypeDefinition(String rawSchema, MediaType mediaType) throws ModelGenerationException {
    if (mediaType == null) {
      if (isXmlSchema(rawSchema)) {
        mediaType = MediaType.APPLICATION_XML_TYPE;
      } else {
        mediaType = MediaType.APPLICATION_JSON_TYPE;
      }
    }

    return TypeDefinition.builder(mediaType,
                                  typeSchemaBuilder.buildTypeSchema(rawSchema),
                                  null,
                                  null)
        .buildObjectTypeDefinition();
  }

  private PrimitiveTypeDefinition.PrimitiveType buildPrimitiveType(APIPrimitiveTypeModel primitiveTypeModel) {
    return primitiveTypeModel.getPrimitiveType();
  }

}
