/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.security;

import static org.mule.connectivity.restconnect.internal.connectormodel.security.APISecurityScheme.SecuritySchemeType.UNSECURED;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.util.HashCodeUtil;

import java.util.LinkedList;
import java.util.List;

public abstract class APISecurityScheme {

  public enum SecuritySchemeType {
    BASIC("Basic Authentication"), OAUTH1("OAuth 1.0"), OAUTH2("OAuth 2.0"), PASS_THROUGH("Pass Through"), DIGEST_AUTHENTICATION(
        "Digest Authentication"), CUSTOM_AUTHENTICATION(
            "Custom Authentication"), UNSECURED("Unsecured"), NOT_SUPPORTED("Not Supported");

    private final String type;

    SecuritySchemeType(String type) {
      this.type = type;
    }

    @Override
    public String toString() {
      return this.type;
    }
  }

  protected final List<Parameter> configParameters = new LinkedList<>();
  protected final List<Parameter> headers = new LinkedList<>();
  protected final List<Parameter> queryParameters = new LinkedList<>();

  protected final SecuritySchemeType schemeType;
  protected final String displayName;

  public APISecurityScheme(SecuritySchemeType schemeType, String displayName) {
    this.schemeType = schemeType;
    this.displayName = displayName;
  }

  public SecuritySchemeType getSecuritySchemeType() {
    return this.schemeType;
  }

  public String getDisplayName() {
    if (schemeType.equals(UNSECURED)) {
      return UNSECURED.name();
    } else {
      return displayName;
    }
  }

  public List<Parameter> getQueryParameters() {
    return queryParameters;
  }

  public List<Parameter> getHeaders() {
    return headers;
  }

  public boolean hasQueryParameters() {
    return !queryParameters.isEmpty();
  }

  public boolean hasHeaders() {
    return !headers.isEmpty();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o)
      return true;
    if (!(o instanceof APISecurityScheme))
      return false;

    APISecurityScheme that = (APISecurityScheme) o;

    return that.hashCode() == this.hashCode();
  }

  @Override
  public int hashCode() {
    return HashCodeUtil.generateHashCode(schemeType, configParameters, headers, queryParameters);
  }

  public static Builder builder(String displayName, List<Parameter> customQueryParams, List<Parameter> customHeaders) {
    return new Builder(displayName, customQueryParams, customHeaders);
  }

  public static class Builder {

    private final List<Parameter> customHeaders;
    private final List<Parameter> customQueryParams;
    private final String displayName;

    public Builder(String displayName, List<Parameter> customQueryParams, List<Parameter> customHeaders) {
      this.customHeaders = customHeaders;
      this.customQueryParams = customQueryParams;
      this.displayName = displayName;
    }

    public BasicAuthScheme buildBasicAuthScheme() {
      return new BasicAuthScheme(displayName);
    }

    public CustomAuthenticationScheme buildCustomAuthenticationScheme() {
      return new CustomAuthenticationScheme(displayName, customQueryParams, customHeaders);
    }

    public DigestAuthenticationScheme buildDigestAuthenticationSchemeScheme() {
      return new DigestAuthenticationScheme(displayName);
    }

    public OAuth2AuthorizationCodeScheme buildOAuth2AuthorizationCodeScheme(String authorizationUri, String accessTokenUri,
                                                                            List<String> scopes) {
      return new OAuth2AuthorizationCodeScheme(displayName,
                                               authorizationUri,
                                               accessTokenUri,
                                               scopes,
                                               customQueryParams,
                                               customHeaders);
    }

    public OAuth2ClientCredentialsScheme buildOAuth2ClientCredentialsScheme(String authorizationUri, String accessTokenUri,
                                                                            List<String> scopes) {
      return new OAuth2ClientCredentialsScheme(displayName,
                                               authorizationUri,
                                               accessTokenUri,
                                               scopes,
                                               customQueryParams,
                                               customHeaders);
    }

    public PassThroughScheme buildPassThroughScheme() {
      return new PassThroughScheme(displayName, customQueryParams, customHeaders);
    }

    public UnsecuredScheme buildUnsecuredScheme() {
      return new UnsecuredScheme();
    }
  }
}
