/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating;

import static java.nio.file.Files.copy;
import static java.nio.file.Files.delete;
import static java.nio.file.Files.find;
import static java.nio.file.Files.write;
import static java.util.stream.Collectors.toList;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.templating.sdk.SdkConnector;
import org.mule.connectivity.restconnect.internal.templating.sdk.SdkPom;
import org.mule.connectivity.restconnect.internal.util.JavaUtils;

import com.google.googlejavaformat.java.Formatter;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.List;
import java.util.stream.Stream;


public class SdkConnectorTemplateEntity extends TemplateEntity {

  private final SdkConnector sdkConnector;
  private final SdkPom sdkPom;
  private final Path projectDir;
  private final Path descriptorPath;

  public SdkConnectorTemplateEntity(Path outputDir, ConnectorModel connectorModel, Path descriptorPath) {
    String projectName = JavaUtils.getJavaUpperCamelNameFromXml(connectorModel.getConnectorXmlName());
    projectDir = outputDir.resolve(projectName);

    this.sdkConnector = new SdkConnector(projectDir, connectorModel);
    this.sdkPom = new SdkPom(projectDir, connectorModel);
    this.descriptorPath = descriptorPath;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    sdkConnector.applyTemplates();
    sdkPom.applyTemplates();
    formatJavaFiles();
    addDescriptorToSources();
  }

  protected void addDescriptorToSources() throws TemplatingException {
    if (descriptorPath != null) {
      try (InputStream descriptorStream = new FileInputStream(descriptorPath.toFile())) {
        copy(descriptorStream,
             sdkConnector.getSourcesPath().resolve("connector-descriptor.yaml"),
             StandardCopyOption.REPLACE_EXISTING);
      } catch (IOException e) {
        throw new TemplatingException("Could not add descriptor to generated sources");
      }
    }
  }

  private void formatJavaFiles() throws TemplatingException {
    for (Path path : getJavaFiles()) {
      javaFormat(path);
    }
  }

  private List<Path> getJavaFiles() throws TemplatingException {
    try (Stream<Path> paths = find(projectDir,
                                   Integer.MAX_VALUE,
                                   (filePath, fileAttr) -> filePath.toString().toLowerCase().endsWith(".java"))) {
      return paths.collect(toList());
    } catch (Exception e) {
      throw new TemplatingException("Could not find files to format", e);
    }
  }

  private void javaFormat(Path filepath) throws TemplatingException {
    String formattedSource;

    try {
      String content = new String(Files.readAllBytes(filepath), StandardCharsets.UTF_8);
      formattedSource = new Formatter().formatSource(content);

      delete(filepath);
      write(filepath, formattedSource.getBytes(StandardCharsets.UTF_8));
    } catch (Exception e) {
      throw new TemplatingException("Could not format file " + filepath.toString(), e);
    }
  }
}
