/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType.HEADER;
import static org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType.QUERY;
import static org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType.URI;
import static org.mule.connectivity.restconnect.internal.util.NamingUtil.isFriendlyName;
import static org.mule.connectivity.restconnect.internal.validation.ModelValidationRules.R013;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.EndPointDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ParameterDescriptor;
import org.mule.connectivity.restconnect.internal.validation.RuleValidator;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;

import java.util.LinkedList;
import java.util.List;

public class R013Validator implements RuleValidator {

  @Override
  public List<ValidationResult> validate(ConnectorDescriptor connectorDescriptor,
                                         APIModel apiModel,
                                         ConnectorModel connectorModel) {

    List<ValidationResult> results = new LinkedList<>();

    for (EndPointDescriptor endpointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endpointDescriptor.getOperations()) {
        if (operationDescriptor.getExpects() != null) {
          results.addAll(validateParameterNames(operationDescriptor.getExpects().getUriParameter(), URI, operationDescriptor,
                                                endpointDescriptor));
          results.addAll(validateParameterNames(operationDescriptor.getExpects().getQueryParameter(), QUERY, operationDescriptor,
                                                endpointDescriptor));
          results.addAll(validateParameterNames(operationDescriptor.getExpects().getHeader(), HEADER, operationDescriptor,
                                                endpointDescriptor));
        }
      }
    }

    return results;
  }

  List<ValidationResult> validateParameterNames(List<ParameterDescriptor> parameters, ParameterType parameterType,
                                                OperationDescriptor operationDescriptor, EndPointDescriptor endpointDescriptor) {
    return parameters.stream()
        .filter(x -> !isFriendlyName(x.getFriendlyName()))
        .map(x -> getValidationError(endpointDescriptor, operationDescriptor, x, parameterType))
        .collect(toList());
  }

  private ValidationResult getValidationError(EndPointDescriptor endpointDescriptor, OperationDescriptor operationDescriptor,
                                              ParameterDescriptor parameterDescriptor, ParameterType parameterType) {
    String location =
        "Operation Descriptor with PATH: " + endpointDescriptor.getPath() +
            " and METHOD: " + operationDescriptor.getMethod() +
            " declares a " + parameterType.getName() + " named " + parameterDescriptor.getFriendlyName() +
            " whose display name is : " + parameterDescriptor.getFriendlyName();

    return new ValidationResult(R013, location);
  }
}
