/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType.HEADER;
import static org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType.QUERY;
import static org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType.URI;
import static org.mule.connectivity.restconnect.internal.validation.ModelValidationRules.R016;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.EndPointDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ParameterDescriptor;
import org.mule.connectivity.restconnect.internal.validation.RuleValidator;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;

import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

public class R016Validator implements RuleValidator {

  @Override
  public List<ValidationResult> validate(ConnectorDescriptor connectorDescriptor,
                                         APIModel apiModel,
                                         ConnectorModel connectorModel) {

    List<ValidationResult> results = new LinkedList<>();

    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        if (operationDescriptor.getExpects() != null) {
          results.addAll(validateParametersExists(operationDescriptor.getExpects().getUriParameter(), URI,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

          results.addAll(validateParametersExists(operationDescriptor.getExpects().getQueryParameter(), QUERY,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

          results.addAll(validateParametersExists(operationDescriptor.getExpects().getHeader(), HEADER,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

        }
      }
    }

    return results;
  }

  private List<ValidationResult> validateParametersExists(List<ParameterDescriptor> parameters,
                                                          ParameterType parameterType,
                                                          String path,
                                                          String method,
                                                          APIModel apiModel) {
    return parameters.stream()
        .map(x -> validateParameterExists(x, parameterType, path, method, apiModel))
        .filter(Objects::nonNull)
        .collect(toList());
  }

  private ValidationResult validateParameterExists(ParameterDescriptor parameter,
                                                   ParameterType parameterType,
                                                   String path,
                                                   String method,
                                                   APIModel apiModel) {

    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path))
        .filter(x -> x.getHttpMethod().name().equalsIgnoreCase(method))
        .findFirst()
        .map(x -> validateParameterExists(x, parameter, parameterType))
        .orElse(null);
  }

  private ValidationResult validateParameterExists(APIOperationModel apiOperation, ParameterDescriptor parameter,
                                                   ParameterType parameterType) {
    List<APIParameterModel> apiParameterList;

    if (parameterType.equals(URI)) {
      apiParameterList = apiOperation.getUriParamsModel();
    } else if (parameterType.equals(QUERY)) {
      apiParameterList = apiOperation.getQueryParamsModel();
    } else {
      apiParameterList = apiOperation.getHeadersModel();
    }

    return apiParameterList.stream().anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameter.getParamName()))
        ? null
        : getValidationError(apiOperation, parameter, parameterType);
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel, ParameterDescriptor parameter,
                                              ParameterType parameterType) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() +
            " and METHOD: " + apiOperationModel.getHttpMethod().name() +
            " does not declare a " + parameterType.getName() + " named " + parameter.getParamName();

    return new ValidationResult(R016, location);
  }
}
