/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser.amf;

import amf.client.model.domain.AnyShape;
import amf.client.model.domain.DomainElement;
import amf.client.model.domain.EndPoint;
import amf.client.model.domain.Operation;
import amf.client.model.domain.Parameter;
import amf.client.model.domain.ParametrizedSecurityScheme;
import amf.client.model.domain.Payload;
import amf.client.model.domain.Response;
import amf.client.model.domain.SecurityScheme;

import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.exception.UnsupportedSecuritySchemeException;
import org.mule.connectivity.restconnect.internal.connectormodel.HTTPMethod;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APISecuritySchemeModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.webapi.util.OperationNamingUtils.buildFriendlyCanonicalOperationName;

public class AMFOperationModel extends APIOperationModel {

  private final EndPoint endPoint;
  private final Operation operation;

  public AMFOperationModel(EndPoint endPoint, Operation operation)
      throws ModelGenerationException {
    this.endPoint = endPoint;
    this.operation = operation;

    this.description = buildOperationDescription();
    this.path = buildOperationPath();
    this.httpMethod = HTTPMethod.fromString(operation.method().value());
    this.name = buildName(this.httpMethod, this.path);
    this.uriParamsModel = buildUriParamsModel();
    this.queryParamsModel = buildQueryParamsModel();
    this.headersModel = buildHeadersModel();
    this.inputMetadataModel = buildInputMetadataModels();
    this.outputMetadataModel = buildOutputMetadataModels();
    this.securitySchemesModel = buildSecuritySchemesModel();
  }

  private List<APISecuritySchemeModel> buildSecuritySchemesModel() throws ModelGenerationException {

    List<SecurityScheme> endPointSchemes =
        endPoint.security().stream().map(this::getScheme).filter(Objects::nonNull).collect(toList());

    List<SecurityScheme> operationSchemes =
        operation.security().stream().map(this::getScheme).filter(Objects::nonNull).collect(toList());

    List<SecurityScheme> selectedSchemes = selectSecuritySchemes(operationSchemes, endPointSchemes, new LinkedList<>());

    List<APISecuritySchemeModel> securitySchemesModel = new ArrayList<>();
    for (SecurityScheme securityScheme : selectedSchemes) {
      securitySchemesModel.add(new AMFSecuritySchemeModel(securityScheme));
    }

    securitySchemesModel = securitySchemesModel.stream().filter(Objects::nonNull).collect(toList());

    // If the operation is secured but we don't support any defined schemes, then we must throw a generation exception.
    if (!selectedSchemes.isEmpty() && securitySchemesModel.isEmpty()) {
      throw new UnsupportedSecuritySchemeException(
                                                   this.httpMethod.toString() + " " + this.path + ": " +
                                                       "None of the specified security schemes are supported.");
    }

    return securitySchemesModel;
  }

  private SecurityScheme getScheme(DomainElement domainElement) {
    if (domainElement instanceof ParametrizedSecurityScheme) {
      ParametrizedSecurityScheme securityScheme = (ParametrizedSecurityScheme) domainElement;
      if (securityScheme.scheme() == null) {
        return null;
      }
      if (securityScheme.scheme().isLink() && securityScheme.scheme().linkTarget().isPresent()) {
        return (SecurityScheme) securityScheme.scheme().linkTarget().orElse(null);
      } else {
        return securityScheme.scheme();
      }
    } else {
      return (SecurityScheme) domainElement;
    }
  }

  private List<APITypeModel> buildOutputMetadataModels() {
    List<APITypeModel> models = new ArrayList<>();
    List<APITypeModel> defaultModel = new ArrayList<>();

    if (operation.responses() != null) {
      for (Response response : operation.responses()) {
        if (response.statusCode().nonEmpty()) {
          if (response.statusCode().value().startsWith("2")) {
            models.addAll(buildAMFTypeModel(response));
          } else if (response.statusCode().value().startsWith("default")) {
            defaultModel.addAll(buildAMFTypeModel(response));
          }
        }
      }
    }

    if (models.isEmpty()) {
      return defaultModel;
    }

    return models;
  }

  private List<APITypeModel> buildAMFTypeModel(Response response) {
    List<APITypeModel> models = new ArrayList<>();
    for (Payload payload : response.payloads()) {
      models.add(new AMFTypeModel((AnyShape) payload.schema(), payload.mediaType().value()));
    }

    return models;
  }

  private List<APITypeModel> buildInputMetadataModels() {
    List<APITypeModel> models = new ArrayList<>();

    if (operation.request() != null) {
      for (Payload payload : operation.request().payloads()) {
        models.add(new AMFTypeModel((AnyShape) payload.schema(), payload.mediaType().value()));
      }
    }

    return models;
  }

  private String buildName(HTTPMethod httpMethod, String path) {

    if (operation.name().nonEmpty() && isNotBlank(operation.name().value())
        && !operation.name().value().equalsIgnoreCase(operation.method().value())) {
      return operation.name().value();
    } else {
      return buildFriendlyCanonicalOperationName(httpMethod, path);
    }
  }

  private String buildOperationPath() {
    return buildStandardPath(endPoint.path().value());
  }

  private String buildOperationDescription() {
    //TODO: RESTC-687: Define if description should be taken from EndPoint or not. This could cause repeated descriptions.
    return operation.description().nonEmpty() ? operation.description().value() : endPoint.description().value();
  }

  private List<APIParameterModel> buildUriParamsModel() {
    List<APIParameterModel> list = new LinkedList<>();

    if (operation.request() != null) {
      for (Parameter x : operation.request().uriParameters()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, ParameterType.URI, false);
        list.add(parameterModel);
      }
    }

    for (Parameter x : endPoint.parameters()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, ParameterType.URI, false);
      list.add(parameterModel);
    }
    return list;
  }

  private List<APIParameterModel> buildQueryParamsModel() {
    List<APIParameterModel> list = new ArrayList<>();
    if (operation.request() != null) {
      for (Parameter x : operation.request().queryParameters()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, ParameterType.QUERY, false);
        list.add(parameterModel);
      }
    }
    return list;
  }

  private List<APIParameterModel> buildHeadersModel() {
    List<APIParameterModel> list = new ArrayList<>();
    if (operation.request() != null) {
      for (Parameter x : operation.request().headers()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, ParameterType.HEADER, false);
        list.add(parameterModel);
      }
    }
    return list;
  }

}
