/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser.amf;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.connectormodel.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.BasicAuthScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.CustomAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.DigestAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.NotSupportedScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2AuthorizationCodeScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2ClientCredentialsScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.PassThroughScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APISecuritySchemeModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.amf.security.AMFOauth2FlowsNaming;
import org.mule.connectivity.restconnect.internal.webapi.parser.amf.security.AMFSecuritySchemesNaming;

import java.util.ArrayList;
import java.util.List;

import amf.client.model.domain.ApiKeySettings;
import amf.client.model.domain.OAuth2Settings;
import amf.client.model.domain.Parameter;
import amf.client.model.domain.SecurityScheme;

public class AMFSecuritySchemeModel extends APISecuritySchemeModel {

  private final SecurityScheme securityScheme;

  public AMFSecuritySchemeModel(SecurityScheme securityScheme) {
    this.securityScheme = securityScheme;

    this.displayName = securityScheme.name().value();
    this.securitySchemeClass = buildSecuritySchemeClass();
    this.customHeaders = buildCustomHeadersModel();
    this.customQueryParams = buildCustomQueryParamsModel();
    this.accessTokenUri = buildAccessTokenUri();
    this.authorizationUri = buildAuthorizationUri();
    this.scopes = buildScopes();
  }

  private String buildAccessTokenUri() {
    if (securityScheme.settings() instanceof OAuth2Settings) {
      OAuth2Settings oAuth2Settings = (OAuth2Settings) securityScheme.settings();
      if (oAuth2Settings != null) {
        return oAuth2Settings.accessTokenUri().value();
      }
    }
    return null;
  }

  private String buildAuthorizationUri() {
    if (securityScheme.settings() instanceof OAuth2Settings) {
      OAuth2Settings oAuth2Settings = (OAuth2Settings) securityScheme.settings();
      if (oAuth2Settings != null) {
        return oAuth2Settings.authorizationUri().value();
      }
    }
    return null;
  }

  private List<String> buildScopes() {
    if (securityScheme.settings() instanceof OAuth2Settings) {
      OAuth2Settings oAuth2Settings = (OAuth2Settings) securityScheme.settings();
      if (oAuth2Settings != null) {
        return oAuth2Settings.scopes().stream().map(s -> s.name().value()).collect(toList());
      }
    }
    return emptyList();
  }

  private Class<? extends APISecurityScheme> buildSecuritySchemeClass() {
    String schemeType = securityScheme.type().value();

    if (AMFSecuritySchemesNaming.isBasicAuth(schemeType)) {
      return BasicAuthScheme.class;
    } else if (AMFSecuritySchemesNaming.isPassThrough(schemeType)) {
      return PassThroughScheme.class;
    } else if (AMFSecuritySchemesNaming.isApiKey(schemeType)) {
      return PassThroughScheme.class;
    } else if (AMFSecuritySchemesNaming.isOauth2(schemeType)) {
      OAuth2Settings oAuth2Settings = (OAuth2Settings) securityScheme.settings();

      if (AMFOauth2FlowsNaming.isAuthorizationCode(oAuth2Settings)) {
        return OAuth2AuthorizationCodeScheme.class;
      } else if (AMFOauth2FlowsNaming.isClientCredentials(oAuth2Settings)) {
        return OAuth2ClientCredentialsScheme.class;
      }
    } else if (AMFSecuritySchemesNaming.isDigest(schemeType)) {
      return DigestAuthenticationScheme.class;
    } else if (AMFSecuritySchemesNaming.isCustom(schemeType)) {
      return CustomAuthenticationScheme.class;
    }

    return NotSupportedScheme.class;
  }

  private List<APIParameterModel> buildCustomQueryParamsModel() {
    List<APIParameterModel> list = new ArrayList<>();
    for (Parameter x : securityScheme.queryParameters()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, ParameterType.QUERY, false);
      list.add(parameterModel);
    }

    AMFParameterModel queryParam = getApiKeyParameter("query");
    if (queryParam != null) {
      list.add(queryParam);
    }

    return list;
  }

  private List<APIParameterModel> buildCustomHeadersModel() {
    List<APIParameterModel> list = new ArrayList<>();
    for (Parameter x : securityScheme.headers()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, ParameterType.HEADER, false);
      list.add(parameterModel);
    }

    AMFParameterModel header = getApiKeyParameter("header");
    if (header != null) {
      list.add(header);
    }

    return list;
  }

  private AMFParameterModel getApiKeyParameter(String from) {
    if (securityScheme.settings() instanceof ApiKeySettings) {
      String in = ((ApiKeySettings) securityScheme.settings()).in().value();
      String name = ((ApiKeySettings) securityScheme.settings()).name().value();
      name = isNotBlank(name) ? name : "api-key";

      if (in.equalsIgnoreCase(from)) {
        return new AMFParameterModel(name, ParameterType.SECURITY, new AMFPrimitiveTypeModel(PrimitiveType.STRING));
      }
    }

    return null;
  }
}
