/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser.amf;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

import amf.client.model.domain.AnyShape;
import amf.client.model.domain.ArrayShape;
import amf.client.model.domain.DataNode;
import amf.client.model.domain.FileShape;
import amf.client.model.domain.NilShape;
import amf.client.model.domain.NodeShape;
import amf.client.model.domain.PropertyShape;
import amf.client.model.domain.ScalarNode;
import amf.client.model.domain.ScalarShape;
import amf.client.model.domain.SchemaShape;
import amf.client.model.domain.Shape;
import amf.client.model.domain.UnionShape;

import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.connectormodel.type.ArrayTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.MultipartTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.EmptyTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.ObjectTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.UnionTypeDefinition;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIPrimitiveTypeModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.amf.util.AMFParserUtil;

import javax.ws.rs.core.MediaType;
import java.util.ArrayList;
import java.util.List;


public class AMFTypeModel extends APITypeModel {

  private AnyShape shape;

  public AMFTypeModel(APIPrimitiveTypeModel primitiveTypeModel) {
    super(primitiveTypeModel);
  }

  public AMFTypeModel(AnyShape anyShape, String mediaType) {
    this.shape = anyShape;

    this.example = buildExample();
    this.enumValues = buildEnumValues();
    this.mediaType = getMediaTypeForStringOrNull(mediaType);
    this.APITypeSchemaModel = new AMFTypeSchemaModel(anyShape, this.mediaType);
    this.typeDefinitionClass = buildTypeDefinitionClass(this.mediaType);
  }

  private Class<? extends TypeDefinition> buildTypeDefinitionClass(MediaType mediaType) {
    Shape actualShape = AMFParserUtil.getActualShape(shape);

    if (actualShape instanceof ScalarShape || actualShape instanceof FileShape) {
      this.primitiveTypeModel = new AMFPrimitiveTypeModel(shape);
      return PrimitiveTypeDefinition.class;
    }

    else if (MediaType.MULTIPART_FORM_DATA_TYPE.equals(mediaType)
        && actualShape instanceof NodeShape
        && !((NodeShape) actualShape).properties().isEmpty()) {
      return MultipartTypeDefinition.class;
    }

    else if (actualShape instanceof NodeShape
        || actualShape instanceof SchemaShape) {
      return ObjectTypeDefinition.class;
    }

    else if (actualShape instanceof ArrayShape) {
      return ArrayTypeDefinition.class;
    }

    else if (actualShape instanceof UnionShape) {
      return UnionTypeDefinition.class;
    }

    else if (actualShape instanceof NilShape) {
      return null;
    }

    else {
      // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
      return EmptyTypeDefinition.class;
    }
  }

  private String buildExample() {
    return !shape.examples().isEmpty() && shape.examples().get(0).value().nonEmpty()
        ? shape.examples().get(0).value().value() : null;
  }

  private List<String> buildEnumValues() {
    return (!shape.values().isEmpty())
        ? shape.values().stream().map(this::getEnumValue).collect(toList()) : null;
  }

  @Override
  public List<APIParameterModel> getParts() {
    if (typeDefinitionClass.equals(MultipartTypeDefinition.class)) {
      List<APIParameterModel> parts = new ArrayList<>();

      for (PropertyShape s : ((NodeShape) shape).properties()) {
        parts.add(new AMFParameterModel(s, ParameterType.PART));
      }

      return parts;
    } else {
      return emptyList();
    }
  }

  @Override
  public APITypeModel getInnerType() {
    if (shape instanceof ArrayShape) {
      return new AMFTypeModel((AnyShape) ((ArrayShape) shape).items(), null);
    } else {
      return null;
    }
  }

  @Override
  public List<APITypeModel> getUnionTypes() {
    if (shape instanceof UnionShape) {
      List<APITypeModel> list = new ArrayList<>();

      for (Shape x : ((UnionShape) shape).anyOf()) {
        AMFTypeModel amfTypeParser = new AMFTypeModel((AnyShape) x, null);
        list.add(amfTypeParser);
      }

      return list;
    } else {
      return emptyList();
    }
  }

  private String getEnumValue(DataNode x) {
    if (x instanceof ScalarNode) {
      return ((ScalarNode) x).value().value();
    }
    throw new IllegalArgumentException("Enum type not supported");
  }
}
