/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser;

import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.exception.UnsupportedSecuritySchemeException;
import org.mule.connectivity.restconnect.internal.connectormodel.HTTPMethod;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APISecuritySchemeModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;

import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.bodies.Response;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;
import org.raml.v2.api.model.v10.methods.Method;
import org.raml.v2.api.model.v10.resources.Resource;
import org.raml.v2.api.model.v10.security.SecurityScheme;
import org.raml.v2.api.model.v10.security.SecuritySchemeRef;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.mule.connectivity.restconnect.internal.webapi.util.OperationNamingUtils.buildFriendlyCanonicalOperationName;

public class RPOperationModel extends APIOperationModel {

  private final Api api;
  private final Method method;

  public RPOperationModel(Api api, Method method) throws ModelGenerationException {
    this.api = api;
    this.method = method;

    this.description = buildDescription();
    this.path = buildPath();
    this.httpMethod = HTTPMethod.fromString(method.method());
    this.name = buildName(this.httpMethod, this.path);
    this.uriParamsModel = buildUriParamsModel();
    this.queryParamsModel = buildQueryParamsModel();
    this.headersModel = buildHeadersModel();
    this.inputMetadataModel = buildInputMetadataModels();
    this.outputMetadataModel = buildOutputMetadataModels();
    this.securitySchemesModel = buildSecuritySchemesModel();
  }

  private String buildDescription() {
    String description = null;

    if (method.description() != null) {
      description = method.description().value();
    } else if (method.resource() != null && requireNonNull(method.resource()).description() != null) {
      description = requireNonNull(method.resource()).description().value();
    }

    return trimToNull(description);
  }

  private String buildPath() {
    return buildStandardPath(requireNonNull(method.resource()).resourcePath());
  }

  private String buildName(HTTPMethod httpMethod, String path) {
    // This checks for the case that the displayName attribute isn't provided, as the parsers
    // defaults this to the method name, per the RAML 1.0 spec.
    if (method.displayName() != null && method.displayName().value() != null
        && !method.displayName().value().equals(method.method())) {
      return trimToNull(method.displayName().value());
    } else {
      return buildFriendlyCanonicalOperationName(httpMethod, path);
    }
  }

  private List<APIParameterModel> buildUriParamsModel() {
    List<APIParameterModel> uriParametersModel = new ArrayList<>();

    // Due to RESTC-34, this is done bottom to the top.
    Resource resource = method.resource();

    while (resource != null) {
      for (APIParameterModel uriParamModel : getParameterList(resource.uriParameters(), ParameterType.URI)) {
        if (uriParametersModel.stream().noneMatch(y -> y.getExternalName().equalsIgnoreCase(uriParamModel.getExternalName()))) {
          uriParametersModel.add(0, uriParamModel);
        }
      }

      resource = resource.parentResource();
    }

    return uriParametersModel;
  }

  private List<APIParameterModel> buildQueryParamsModel() {
    return getParameterList(method.queryParameters(), ParameterType.QUERY);
  }

  private List<APIParameterModel> buildHeadersModel() {
    return getParameterList(method.headers(), ParameterType.HEADER);
  }

  private List<APIParameterModel> getParameterList(List<TypeDeclaration> typeDeclarations, ParameterType parameterType) {
    List<APIParameterModel> parametersModel = new LinkedList<>();

    for (TypeDeclaration td : typeDeclarations) {
      parametersModel.add(new RPParameterModel(td, parameterType, false));
    }

    return parametersModel;
  }

  private List<APITypeModel> buildInputMetadataModels() {
    List<APITypeModel> typeModels = new LinkedList<>();

    for (TypeDeclaration td : method.body()) {
      typeModels.add(new RPTypeModel(td, td.name()));
    }

    return typeModels;
  }

  private List<APITypeModel> buildOutputMetadataModels() {
    List<Response> possibleResponses =
        method.responses().stream().filter(x -> hasBody(x) && isOkResponse(x)).collect(toList());

    List<APITypeModel> typeModels = new LinkedList<>();

    for (Response response : possibleResponses) {
      for (TypeDeclaration td : response.body()) {
        typeModels.add(new RPTypeModel(td, td.name()));
      }
    }

    return typeModels;
  }

  private boolean hasBody(Response response) {
    return response.body() != null && !response.body().isEmpty();
  }

  private boolean isOkResponse(Response response) {
    return response.code().value().startsWith("2");
  }

  private List<APISecuritySchemeModel> buildSecuritySchemesModel() throws ModelGenerationException {

    List<SecurityScheme> globalSchemes =
        api.securedBy().stream()
            .filter(Objects::nonNull)
            .map(SecuritySchemeRef::securityScheme)
            .filter(Objects::nonNull)
            .collect(toList());

    List<SecurityScheme> endPointSchemes =
        method.resource().securedBy().stream()
            .filter(Objects::nonNull)
            .map(SecuritySchemeRef::securityScheme)
            .filter(Objects::nonNull)
            .collect(toList());

    List<SecurityScheme> methodSchemes =
        method.securedBy().stream()
            .filter(Objects::nonNull)
            .map(SecuritySchemeRef::securityScheme)
            .filter(Objects::nonNull)
            .collect(toList());

    List<SecurityScheme> selectedSchemes = selectSecuritySchemes(methodSchemes, endPointSchemes, globalSchemes);

    List<APISecuritySchemeModel> securitySchemesModel = new ArrayList<>();
    if (!selectedSchemes.isEmpty()) {
      for (SecurityScheme securityScheme : selectedSchemes) {
        securitySchemesModel.add(new RPSecuritySchemeModel(securityScheme));
      }
    }

    securitySchemesModel = securitySchemesModel.stream().filter(Objects::nonNull).collect(toList());

    // If the operation is secured but we don't support any defined schemes, then we must throw a generation exception.
    if (!selectedSchemes.isEmpty() && securitySchemesModel.isEmpty()) {
      throw new UnsupportedSecuritySchemeException(
                                                   this.httpMethod.toString() + " " + this.path + ": " +
                                                       "None of the specified security schemes are supported.");
    }

    return securitySchemesModel;
  }

}
