/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser;

import static java.util.Collections.emptyList;
import static org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser.util.RamlParserUtils.getValueFromAnnotableString;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.connectormodel.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.BasicAuthScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.CustomAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.DigestAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.NotSupportedScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2AuthorizationCodeScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2ClientCredentialsScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.PassThroughScheme;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APISecuritySchemeModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser.security.RPOauth2FlowsNaming;
import org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser.security.RPSecuritySchemesNaming;

import java.util.ArrayList;
import java.util.List;

import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;
import org.raml.v2.api.model.v10.security.SecurityScheme;

public class RPSecuritySchemeModel extends APISecuritySchemeModel {

  private final SecurityScheme securityScheme;

  public RPSecuritySchemeModel(SecurityScheme securityScheme) {
    this.securityScheme = securityScheme;

    this.displayName = securityScheme.name();
    this.securitySchemeClass = buildSecuritySchemeClass();
    this.customHeaders = buildCustomHeadersModel();
    this.customQueryParams = buildCustomQueryParamsModel();
    this.accessTokenUri = buildAccessTokenUri();
    this.authorizationUri = buildAuthorizationTokenUri();
    this.scopes = buildScopes();
  }

  private List<String> buildScopes() {
    if (securityScheme.settings() != null) {
      return securityScheme.settings().scopes();
    }
    return emptyList();
  }

  private String buildAuthorizationTokenUri() {
    if (securityScheme.settings() != null) {
      return getValueFromAnnotableString(securityScheme.settings().authorizationUri());
    }
    return null;
  }

  private String buildAccessTokenUri() {
    if (securityScheme.settings() != null) {
      return getValueFromAnnotableString(securityScheme.settings().accessTokenUri());
    }
    return null;
  }

  private Class<? extends APISecurityScheme> buildSecuritySchemeClass() {

    String schemeType = securityScheme.type();

    if (RPSecuritySchemesNaming.isBasicAuth(schemeType)) {
      return BasicAuthScheme.class;
    } else if (RPSecuritySchemesNaming.isPassThrough(schemeType)) {
      return PassThroughScheme.class;
    } else if (RPSecuritySchemesNaming.isOauth2(schemeType)) {
      for (String grant : securityScheme.settings().authorizationGrants()) {
        if (RPOauth2FlowsNaming.isAuthorizationCode(grant)) {
          return OAuth2AuthorizationCodeScheme.class;
        } else if (RPOauth2FlowsNaming.isClientCredentials(grant)) {
          return OAuth2ClientCredentialsScheme.class;
        }
      }
    } else if (RPSecuritySchemesNaming.isDigest(schemeType)) {
      return DigestAuthenticationScheme.class;
    } else if (RPSecuritySchemesNaming.isCustom(schemeType)) {
      // If none of the previous types matches, it is a custom security scheme
      // https://github.com/raml-org/raml-spec/blob/master/versions/raml-10/raml-10.md#security-scheme-declaration
      return CustomAuthenticationScheme.class;
    }

    return NotSupportedScheme.class;
  }

  private List<APIParameterModel> buildCustomQueryParamsModel() {
    if (securityScheme.describedBy() != null) {
      return getParameterList(securityScheme.describedBy().queryParameters(), ParameterType.QUERY);
    }
    return emptyList();
  }

  private List<APIParameterModel> buildCustomHeadersModel() {
    if (securityScheme.describedBy() != null) {
      return getParameterList(securityScheme.describedBy().headers(), ParameterType.HEADER);
    }
    return emptyList();
  }

  private List<APIParameterModel> getParameterList(List<TypeDeclaration> typeDeclarations, ParameterType parameterType) {
    List<APIParameterModel> parametersModel = new ArrayList<>();

    for (TypeDeclaration td : typeDeclarations) {
      parametersModel.add(new RPParameterModel(td, parameterType, false));
    }

    return parametersModel;
  }
}
