/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static java.util.stream.Collectors.toList;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.security.ConnectorSecurityScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.BasicAuthScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.CustomAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.DigestAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.NotSupportedScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2AuthorizationCodeScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2ClientCredentialsScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.PassThroughScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.UnsecuredScheme;
import org.mule.connectivity.restconnect.internal.descriptor.model.SecuritySchemeDescriptor;
import org.mule.connectivity.restconnect.internal.webapi.model.APISecuritySchemeModel;

import java.util.LinkedList;
import java.util.List;

public class ConnectorSecuritySchemeBuilder {

  private final ConnectorParameterBuilder parameterBuilder;

  public ConnectorSecuritySchemeBuilder(ConnectorTypeDefinitionBuilder connectorTypeDefinitionBuilder) {
    this.parameterBuilder = new ConnectorParameterBuilder(connectorTypeDefinitionBuilder);
  }

  public List<ConnectorSecurityScheme> buildSecuritySchemes(List<APISecuritySchemeModel> apiSecuritySchemeModels,
                                                            List<SecuritySchemeDescriptor> securities)
      throws ModelGenerationException {
    List<ConnectorSecurityScheme> apiSecuritySchemes = new LinkedList<>();

    if (!apiSecuritySchemeModels.isEmpty()) {
      for (APISecuritySchemeModel securitySchemeModel : apiSecuritySchemeModels) {
        ConnectorSecurityScheme apiSecurityScheme =
            buildSecurityScheme(securitySchemeModel, getSecuritySchemeDescriptor(securities, securitySchemeModel));
        if (apiSecurityScheme == null) {
          continue;
        }
        if (!(apiSecurityScheme instanceof NotSupportedScheme)) {
          apiSecuritySchemes.add(apiSecurityScheme);
        }
      }
    } else {
      apiSecuritySchemes.add(new UnsecuredScheme());
    }

    return apiSecuritySchemes;
  }

  private SecuritySchemeDescriptor getSecuritySchemeDescriptor(List<SecuritySchemeDescriptor> securities,
                                                               APISecuritySchemeModel securitySchemeModel) {
    return securities.stream()
        .filter(x -> x.getDisplayName().equalsIgnoreCase(securitySchemeModel.getName()))
        .findFirst().orElse(null);
  }


  private ConnectorSecurityScheme buildSecurityScheme(APISecuritySchemeModel securitySchemeModel,
                                                      SecuritySchemeDescriptor securitySchemeDescriptor)
      throws ModelGenerationException {
    if (securitySchemeDescriptor != null && securitySchemeDescriptor.isIgnored()) {
      return null;
    }
    Class<?> securitySchemeClass = securitySchemeModel.getSecuritySchemeClass();
    List<String> collisionNames = new LinkedList<>();

    List<Parameter> customQueryParams =
        parameterBuilder.buildParameterList(securitySchemeModel.getCustomQueryParams(), securitySchemeDescriptor, collisionNames);
    collisionNames.addAll(customQueryParams.stream().map(Parameter::getInternalName).collect(toList()));
    List<Parameter> customHeaders =
        parameterBuilder.buildParameterList(securitySchemeModel.getCustomHeaders(), securitySchemeDescriptor, collisionNames);

    ConnectorSecurityScheme.Builder builder =
        ConnectorSecurityScheme.builder(securitySchemeModel.getName(), customQueryParams, customHeaders);

    if (securitySchemeClass.equals(BasicAuthScheme.class)) {
      return builder.buildBasicAuthScheme();
    } else if (securitySchemeClass.equals(CustomAuthenticationScheme.class)) {
      return builder.buildCustomAuthenticationScheme();
    } else if (securitySchemeClass.equals(DigestAuthenticationScheme.class)) {
      return builder.buildDigestAuthenticationSchemeScheme();
    } else if (securitySchemeClass.equals(OAuth2AuthorizationCodeScheme.class)) {
      return builder.buildOAuth2AuthorizationCodeScheme(securitySchemeModel.getAuthorizationUri(),
                                                        securitySchemeModel.getAccessTokenUri(),
                                                        securitySchemeModel.getScopes());
    } else if (securitySchemeClass.equals(OAuth2ClientCredentialsScheme.class)) {
      return builder.buildOAuth2ClientCredentialsScheme(securitySchemeModel.getAuthorizationUri(),
                                                        securitySchemeModel.getAccessTokenUri(),
                                                        securitySchemeModel.getScopes());
    } else if (securitySchemeClass.equals(PassThroughScheme.class)) {
      return builder.buildPassThroughScheme();
    } else {
      return builder.buildUnsecuredScheme();
    }
  }
}

