/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.Pagination;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.JsonTypeSchema;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.XmlTypeSchema;
import org.mule.connectivity.restconnect.internal.templating.JavaTemplateEntity;
import org.mule.connectors.restconnect.commons.api.configuration.RestConnectConfiguration;
import org.mule.connectors.restconnect.commons.api.connection.RestConnection;
import org.mule.connectors.restconnect.commons.api.error.RequestErrorTypeProvider;
import org.mule.connectors.restconnect.commons.api.operation.BaseRestConnectOperation;
import org.mule.connectors.restconnect.commons.api.operation.ConfigurationOverrides;
import org.mule.connectors.restconnect.commons.api.operation.EntityRequestParameters;
import org.mule.connectors.restconnect.commons.api.operation.HttpResponseAttributes;
import org.mule.connectors.restconnect.commons.api.operation.NonEntityRequestParameters;
import org.mule.connectors.restconnect.commons.internal.RestConnectConstants;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.metadata.fixed.OutputJsonType;
import org.mule.runtime.extension.api.annotation.metadata.fixed.OutputXmlType;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import javax.lang.model.element.Modifier;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import static com.google.common.base.CaseFormat.LOWER_CAMEL;
import static com.google.common.base.CaseFormat.UPPER_UNDERSCORE;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.abbreviateText;

public abstract class AbstractSdkOperation extends JavaTemplateEntity {

  public static final String PARAM_DOC_NAME_DESCRIPTION = "@param $L $L\n";

  public static final String CONTENT_TYPE_HEADER_NAME = "content-type";
  public static final String ACCEPT_HEADER_NAME = "accept";

  public static final String ADD_QUERY_PARAM_METHOD_NAME = "addQueryParam";
  public static final String ADD_HEADER_METHOD_NAME = "addHeader";

  public static final String ADD_MULTIPLE_QUERY_PARAM_METHOD_NAME = "queryParams";
  public static final String ADD_MULTIPLE_HEADER_METHOD_NAME = "headers";

  private static final String REQUEST_PARAMETERS_GROUP_NAME = "REQUEST_PARAMETERS_GROUP_NAME";
  private static final String CONNECTOR_OVERRIDES = "CONNECTOR_OVERRIDES";

  private final ConnectorOperation operation;
  protected final List<SdkParameter> allPathParameters;
  protected final List<SdkParameter> allQueryParameters;
  protected final List<SdkParameter> allHeaders;
  protected final SdkContent content;

  protected final SdkTypeDefinition sdkOutputMetadata;

  private final SdkMtfOperationTest sdkMtfOperationTest;

  public abstract FieldSpec generateExpressionLanguageField();

  public abstract ParameterSpec generateInitialPagingParameter();

  public abstract TypeName generateMethodReturn();

  public abstract CodeBlock generateOperationMethodBody();

  public AbstractSdkOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                              ConnectorOperation operation) {
    super(outputDir, connectorModel);

    this.operation = operation;

    this.allPathParameters =
        operation.getUriParameters().stream()
            .map(x -> new SdkParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(), x)).collect(toList());
    this.allQueryParameters =
        operation.getQueryParameters().stream()
            .map(x -> new SdkParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(), x)).collect(toList());
    this.allHeaders =
        operation.getHeaders().stream().map(x -> new SdkParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(), x))
            .collect(toList());

    this.content = operation.getInputMetadata() != null
        ? new SdkContent(outputDir, connectorModel, sdkConnector, operation)
        : null;

    this.sdkOutputMetadata = operation.getOutputMetadata() != null
        ? new SdkTypeDefinition(sdkConnector, operation, false)
        : null;

    //create mtf
    this.sdkMtfOperationTest = new SdkMtfOperationTest(operation, outputDir);

  }

  public SdkTypeDefinition getSdkOutputMetadata() {
    return sdkOutputMetadata;
  }

  public String getJavaClassName() {
    return getJavaUpperCamelNameFromXml(operation.getInternalName()) + "Operation";
  }

  private String getJavaMethodName() {
    return getJavaLowerCamelNameFromXml(operation.getInternalName());
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + ".internal.operation";
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (content != null) {
      content.applyTemplates();
    }

    for (SdkParameter sdkParameter : allPathParameters) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : allQueryParameters) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : allHeaders) {
      sdkParameter.applyTemplates();
    }

    sdkMtfOperationTest.applyTemplates();

    generateOperationClass();
  }

  protected void generateOperationClass() throws TemplatingException {
    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(Modifier.PUBLIC)
            .superclass(BaseRestConnectOperation.class)
            .addMethod(generateOperationMethod());

    FieldSpec expressionLanguage = generateExpressionLanguageField();
    if (expressionLanguage != null) {
      operationClassBuilder.addField(expressionLanguage);
    }
    for (SdkParameter pathParam : this.allPathParameters) {
      operationClassBuilder.addField(generatePathParamPatternField(pathParam));
    }

    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getPackage(), operationClassBuilder.build())
        .addStaticImport(RestConnectConstants.class, REQUEST_PARAMETERS_GROUP_NAME, CONNECTOR_OVERRIDES);

    if (!this.allPathParameters.isEmpty()) {
      javaFileBuilder.addStaticImport(Pattern.class, "compile");
    }

    writeClassToFile(javaFileBuilder.build());
  }

  private FieldSpec generatePathParamPatternField(SdkParameter pathParam) {
    return FieldSpec.builder(Pattern.class,
                             getPathParamPatternFieldName(pathParam),
                             Modifier.PRIVATE,
                             Modifier.FINAL,
                             Modifier.STATIC)
        .initializer("compile(\"\\\\{$L}\")", pathParam.getExternalName())
        .build();
  }

  private String getPathParamPatternFieldName(SdkParameter pathParam) {
    return LOWER_CAMEL.to(UPPER_UNDERSCORE, pathParam.getJavaName()) + "_PATTERN";
  }

  private AnnotationSpec generateDescriptionAnnotation() {
    return AnnotationSpec
        .builder(Summary.class)
        .addMember(VALUE_MEMBER, "$S", abbreviateText(operation.getDescription()))
        .build();
  }

  public MethodSpec generateOperationMethod() {
    CodeBlock.Builder javaDoc = CodeBlock.builder()
        .add("\n$L\n", defaultIfEmpty(operation.getDescription(), operation.getDisplayName()))
        .add("\nThis operation makes an HTTP $L request to the $L endpoint", operation.getHttpMethod().toUpperCase(),
             operation.getPath())
        .add("\n");

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(Modifier.PUBLIC);

    if (requiresConfigParameter()) {
      ParameterSpec configParameter = generateConfigParameter();
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, configParameter.name, "the configuration to use");
      methodBuilder.addParameter(configParameter);
    }

    if (requiresConnectionParameter()) {
      ParameterSpec connectionParameter = generateConnectionParameter();
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, connectionParameter.name, "the connection to use");
      methodBuilder.addParameter(connectionParameter);
    }

    for (SdkParameter sdkParam : allPathParameters) {
      ParameterSpec parameterSpec = sdkParam.generateParameterParameter().build();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                  defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
    }

    for (SdkParameter sdkParam : allQueryParameters) {
      if (!isQueryParamDefinedInPagination(sdkParam.getExternalName())) {
        ParameterSpec parameterSpec = sdkParam.generateParameterParameter().build();
        methodBuilder.addParameter(parameterSpec);
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                    defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
      }
    }

    for (SdkParameter sdkParam : allHeaders) {
      ParameterSpec parameterSpec = sdkParam.generateParameterParameter().build();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                  defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
    }

    ParameterSpec initialPagingParameter = generateInitialPagingParameter();
    if (initialPagingParameter != null) {
      methodBuilder.addParameter(initialPagingParameter);
      Optional<AnnotationSpec> summary =
          initialPagingParameter.annotations.stream().filter(x -> x.members.containsKey(VALUE_MEMBER)).findFirst();
      if (summary.isPresent()) {
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, initialPagingParameter.name,
                    summary.get().members.get(VALUE_MEMBER).get(0).toString().replaceAll("\"", ""));
      }
    }

    if (content != null) {
      ParameterSpec parameterSpec = content.generateContentParameter();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name, "the content to use");
    }

    ParameterSpec parameterSpec = generateRequestParametersParameter();
    methodBuilder.addParameter(parameterSpec);

    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                "the {@link " + ((ClassName) parameterSpec.type).simpleName() + "}");

    ParameterSpec configurationOverridesParameter = generateConfigurationOverridesParameter();
    methodBuilder.addParameter(configurationOverridesParameter);
    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, configurationOverridesParameter.name, "the {@link ConfigurationOverrides}");

    ParameterSpec streamingHelperParameter = generateStreamingHelperParameter();
    methodBuilder.addParameter(streamingHelperParameter);
    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, streamingHelperParameter.name, "the {@link StreamingHelper}");

    if (requiresCallbackParameter()) {
      ParameterSpec completionCallbackParameter = generateCompletionCallbackParameter();
      methodBuilder.addParameter(completionCallbackParameter);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, completionCallbackParameter.name, "the operation's {@link CompletionCallback}");
    }

    methodBuilder.addAnnotation(generateThrowsAnnotation());

    methodBuilder.addAnnotation(generateDisplayNameAnnotation());

    if (isNotBlank(operation.getDescription())) {
      methodBuilder.addAnnotation(generateDescriptionAnnotation());
    }

    if (requiresMediaTypeAnnotation()) {
      if (operation.getOutputMetadata() != null) {
        methodBuilder.addAnnotation(generateMediaTypeAnnotation());
      } else {
        methodBuilder.addAnnotation(generateDefaultMediaTypeAnnotation());
      }
    }

    addOutputTypeAnnotationToMethod(methodBuilder);

    methodBuilder.addCode(generateOperationMethodBody());
    methodBuilder.addJavadoc(javaDoc.build());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  protected void addOutputTypeAnnotationToMethod(MethodSpec.Builder methodBuilder) {
    if (sdkOutputMetadata != null && sdkOutputMetadata.getJavaType().equals(InputStream.class)) {
      if (operation.getOutputMetadata().getTypeSchema() instanceof XmlTypeSchema) {
        methodBuilder.addAnnotation(generateOutputXmlTypeAnnotation());
      } else if (operation.getOutputMetadata().getTypeSchema() instanceof JsonTypeSchema) {
        methodBuilder.addAnnotation(generateOutputJsonTypeAnnotation());
      }
    }
  }

  private boolean isQueryParamDefinedInPagination(String paramName) {
    Pagination pagination = this.connectorModel.getPagination(this.operation.getPagination());
    if (pagination != null) {
      return pagination.containsParameterExternalName(paramName);
    }

    return false;
  }

  private AnnotationSpec generateDisplayNameAnnotation() {
    return AnnotationSpec.builder(DisplayName.class)
        .addMember(VALUE_MEMBER, "$S", operation.getDisplayName())
        .build();
  }

  private AnnotationSpec generateOutputJsonTypeAnnotation() {
    return AnnotationSpec
        .builder(OutputJsonType.class)
        .addMember("schema", "$S", getOutputMetadataSchemaPath())
        .build();
  }

  private AnnotationSpec generateOutputXmlTypeAnnotation() {
    return AnnotationSpec
        .builder(OutputXmlType.class)
        .addMember("schema", "$S", getOutputMetadataSchemaPath())
        .build();
  }

  private String getOutputMetadataSchemaPath() {
    String schemaPath = sdkOutputMetadata.getSchemaPath();
    if (schemaPath.startsWith("/")) {
      schemaPath = schemaPath.substring(1);
    }
    return schemaPath;
  }

  public CodeBlock.Builder generateCommonOperationMethodBody() {
    CodeBlock.Builder methodBody = CodeBlock.builder();

    methodBody.addStatement("$T requestPath = $S", String.class, operation.getPath());

    for (SdkParameter pathParam : allPathParameters) {
      if (pathParam.isArrayType()) {
        methodBody
            .addStatement("requestPath = $L.matcher(requestPath).replaceAll($L.stream().map(v -> $L).collect($T.joining(\",\")))",
                          getPathParamPatternFieldName(pathParam),
                          pathParam.getJavaName(),
                          pathParam.getStringValueGetter("v"),
                          Collectors.class);
      } else {
        methodBody.addStatement("requestPath = $L.matcher(requestPath).replaceAll($L)",
                                getPathParamPatternFieldName(pathParam),
                                pathParam.getStringValueGetter());
      }
    }

    String baseUriString = "connection.getBaseUri()";

    if (isNotBlank(operation.getAlternativeBaseUri()) && connectorModel.getBaseUri().isMultipleBaseUri()) {
      String multipleBaseUri =
          connectorModel.getBaseUri().getMultipleBaseUriOrDefault(operation.getAlternativeBaseUri());

      baseUriString = "\"" + multipleBaseUri + "\"";
    }

    methodBody.addStatement("$T builder = requestBuilder($L, requestPath, $T.$L, $L, parameters)",
                            HttpRequestBuilder.class,
                            baseUriString,
                            HttpConstants.Method.class,
                            operation.getHttpMethod().toUpperCase(),
                            content != null ? "content" : "null");

    if (operation.getInputMetadata() != null && operation.getInputMetadata().getMediaType() != null) {
      methodBody.addStatement("builder.$L($S, $S)",
                              ADD_HEADER_METHOD_NAME,
                              CONTENT_TYPE_HEADER_NAME,
                              operation.getInputMetadata().getMediaType().toString());
    }

    if (operation.getOutputMetadata() != null && operation.getOutputMetadata().getMediaType() != null) {
      methodBody.addStatement("builder.$L($S, $S)",
                              ADD_HEADER_METHOD_NAME,
                              ACCEPT_HEADER_NAME,
                              operation.getOutputMetadata().getMediaType().toString());
    }

    for (SdkParameter queryParam : allQueryParameters) {
      if (!isQueryParamDefinedInPagination(queryParam.getExternalName())) {
        methodBody.add(generateRequestBuilderParameterCodeBlock(queryParam,
                                                                ADD_QUERY_PARAM_METHOD_NAME,
                                                                ADD_MULTIPLE_QUERY_PARAM_METHOD_NAME));
      }
    }

    for (SdkParameter header : allHeaders) {
      methodBody.add(generateRequestBuilderParameterCodeBlock(header,
                                                              ADD_HEADER_METHOD_NAME,
                                                              ADD_MULTIPLE_HEADER_METHOD_NAME));
    }

    return methodBody;
  }

  private CodeBlock generateRequestBuilderParameterCodeBlock(SdkParameter parameter, String addSingleValueMethodName,
                                                             String addMultipleValueMethodName) {
    CodeBlock.Builder builder = CodeBlock.builder();
    if (parameter.isArrayType()) {
      builder.addStatement("$T<$T, $T> $LParams = new $T<>()",
                           MultiMap.class, String.class, String.class,
                           parameter.getJavaName(),
                           MultiMap.class);

      builder.addStatement("$L.stream().filter(v -> v != null).forEach(v -> $LParams.put($S, $L))",
                           parameter.getJavaName(), parameter.getJavaName(), parameter.getExternalName(),
                           parameter.getStringValueGetter("v"));

      builder.addStatement("builder.$L($LParams)", addMultipleValueMethodName, parameter.getJavaName());
    } else {
      if (parameter.isNullable()) {
        builder.beginControlFlow("if ($L != null)", parameter.getJavaName());
      }
      builder.addStatement("builder.$L($S, $L)", addSingleValueMethodName,
                           parameter.getExternalName(), parameter.getStringValueGetter());

      if (parameter.isNullable()) {
        builder.endControlFlow();
      }
    }

    return builder.build();
  }


  private AnnotationSpec generateThrowsAnnotation() {
    return AnnotationSpec
        .builder(Throws.class)
        .addMember(VALUE_MEMBER, "$T.class", RequestErrorTypeProvider.class)
        .build();
  }

  private AnnotationSpec generateMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", operation.getOutputMetadata().getMediaType())
        .build();
  }

  private AnnotationSpec generateDefaultMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", operationMethodRequiresBody() ? "application/json" : "text/plain")
        .build();
  }

  private ParameterSpec generateConfigParameter() {
    return ParameterSpec
        .builder(RestConnectConfiguration.class, "config")
        .addAnnotation(Config.class)
        .build();
  }

  private ParameterSpec generateConnectionParameter() {
    return ParameterSpec
        .builder(RestConnection.class, "connection")
        .addAnnotation(Connection.class)
        .build();
  }

  private ParameterSpec generateRequestParametersParameter() {
    AnnotationSpec parameterGroupAnnotation =
        AnnotationSpec
            .builder(ParameterGroup.class)
            .addMember(NAME_MEMBER, REQUEST_PARAMETERS_GROUP_NAME)
            .build();

    Class<?> requestParameterClass =
        operation.getInputMetadata() != null ? EntityRequestParameters.class : NonEntityRequestParameters.class;

    return ParameterSpec
        .builder(requestParameterClass, "parameters")
        .addAnnotation(parameterGroupAnnotation)
        .build();
  }

  private ParameterSpec generateConfigurationOverridesParameter() {
    AnnotationSpec parameterGroupAnnotation =
        AnnotationSpec
            .builder(ParameterGroup.class)
            .addMember(NAME_MEMBER, CONNECTOR_OVERRIDES)
            .build();

    return ParameterSpec
        .builder(ConfigurationOverrides.class, "overrides")
        .addAnnotation(parameterGroupAnnotation)
        .build();
  }

  private ParameterSpec generateStreamingHelperParameter() {
    return ParameterSpec
        .builder(StreamingHelper.class, "streamingHelper")
        .build();
  }

  private ParameterSpec generateCompletionCallbackParameter() {
    if (!isVoidOperation() && (getSdkOutputMetadata() != null || operationMethodRequiresBody())) {
      return ParameterSpec
          .builder(ParameterizedTypeName.get(CompletionCallback.class, InputStream.class, HttpResponseAttributes.class),
                   "callback")
          .build();
    } else {
      return ParameterSpec
          .builder(ParameterizedTypeName.get(CompletionCallback.class, String.class, HttpResponseAttributes.class), "callback")
          .build();
    }
  }

  protected boolean operationMethodRequiresBody() {
    return operation.getHttpMethod().equalsIgnoreCase("get")
        || operation.getHttpMethod().equalsIgnoreCase("post")
        || operation.getHttpMethod().equalsIgnoreCase("patch")
        || operation.getHttpMethod().equalsIgnoreCase("options");
  }

  protected boolean isVoidOperation() {
    return operation.getVoidOperation() != null && operation.getVoidOperation();
  }

  protected boolean requiresConnectionParameter() {
    return true;
  }

  protected boolean requiresCallbackParameter() {
    return true;
  }

  protected boolean requiresConfigParameter() {
    return false;
  }

  protected boolean requiresMediaTypeAnnotation() {
    return true;
  }

}
