/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.InPaginationParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.OutPaginationParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.Pagination;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.PaginationParameter;
import org.mule.connectors.restconnect.commons.api.connection.RestConnection;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.metadata.OutputResolver;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import javax.inject.Inject;
import javax.lang.model.element.Modifier;
import java.nio.file.Path;
import java.util.function.Function;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;

import static org.mule.connectivity.restconnect.internal.util.JavaUtils.abbreviateText;


public abstract class AbstractSdkPaginationOperation extends AbstractSdkOperation implements SdkPaginationStratety {

  public static final String PAGING_RESPONSE_EXPRESSION = "pagingResponseExpression";

  private final Pagination pagination;
  private final SdkPagingTypeResolver pagingTypeResolver;

  public abstract String getInitialPagingParameterSummary();

  public AbstractSdkPaginationOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                        ConnectorOperation operation) {
    super(outputDir, connectorModel, sdkConnector, operation);
    this.pagination = connectorModel.getPagination(operation.getPagination());
    this.pagingTypeResolver =
        new SdkPagingTypeResolver(outputDir, connectorModel, operation, operation.getOutputMetadata(), sdkOutputMetadata);
  }

  public Pagination getPagination() {
    return pagination;
  }

  public AnnotationSpec buildSummaryAnnotation() {
    return AnnotationSpec
        .builder(Summary.class)
        .addMember(VALUE_MEMBER, "$S", abbreviateText(getInitialPagingParameterSummary()))
        .build();
  }

  public AnnotationSpec buildDefaultAnnotation() {
    return AnnotationSpec.builder(Optional.class)
        .addMember("defaultValue", "$S", getInitialValue())
        .build();
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    pagingTypeResolver.applyTemplates();

    super.applyTemplates();
  }

  @Override
  public ParameterSpec generateInitialPagingParameter() {
    return ParameterSpec
        .builder(TypeName.INT, getInitialParamName())
        .addAnnotation(buildDefaultAnnotation())
        .addAnnotation(buildSummaryAnnotation())
        .build();
  }

  @Override
  public CodeBlock getPagingMethodOperation() {
    CodeBlock.Builder paginationBody = CodeBlock.builder();
    paginationBody.addStatement(
                                "return new $T($S, $L, requestFactory, expressionLanguage, streamingHelper, $S, org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON, config.getEncoding(), overrides.getResponseTimeoutAsMillis())",
                                getPagingProvider(),
                                getTokenParamName(),
                                generateInitialPagingParameter().name,
                                getPayloadExpression());
    return paginationBody.build();
  }

  @Override
  public TypeName generateMethodReturn() {
    return ParameterizedTypeName.get(ClassName.get(PagingProvider.class),
                                     TypeName.get(RestConnection.class),
                                     ParameterizedTypeName.get(TypedValue.class, String.class));
  }

  @Override
  public CodeBlock generateOperationMethodBody() {
    CodeBlock.Builder paginationBody = CodeBlock.builder();

    CodeBlock.Builder methodBody = super.generateCommonOperationMethodBody();
    methodBody.addStatement("return builder");
    paginationBody.addStatement("$T<$T,$T> requestFactory = connection -> { $L } ",
                                Function.class,
                                RestConnection.class,
                                HttpRequestBuilder.class,
                                methodBody.build().toString());

    paginationBody.add(getPagingMethodOperation());

    return paginationBody.build();
  }

  @Override
  protected void addOutputTypeAnnotationToMethod(MethodSpec.Builder methodBuilder) {
    if (pagingTypeResolver.getRequiresTypeResolver()) {
      AnnotationSpec outputAnnotation = AnnotationSpec
          .builder(OutputResolver.class)
          .addMember("output", "$T.class", ClassName.get(pagingTypeResolver.getPackage(), pagingTypeResolver.getClassName()))
          .build();

      methodBuilder.addAnnotation(outputAnnotation);
    }
  }

  @Override
  public FieldSpec generateExpressionLanguageField() {
    return FieldSpec
        .builder(ExpressionLanguage.class, "expressionLanguage", Modifier.PRIVATE)
        .addAnnotation(AnnotationSpec.builder(Inject.class).build())
        .build();
  }

  @Override
  public String getInitialValue() {
    PaginationParameter parameter = getPagination().getParameter(getInitialParamName());
    if (parameter == null) {
      return "0";
    }

    return (parameter instanceof InPaginationParameter)
        ? ((InPaginationParameter) parameter).getValueExtraction().getContent()
        : ((OutPaginationParameter) parameter).getValue();
  }

  @Override
  public String getPayloadExpression() {
    PaginationParameter parameter = this.pagination.getParameter(PAGING_RESPONSE_EXPRESSION);

    return (parameter == null) ? null : ((InPaginationParameter) parameter).getValueExtraction().getContent();
  }

  @Override
  protected boolean requiresConnectionParameter() {
    return false;
  }

  @Override
  protected boolean requiresCallbackParameter() {
    return false;
  }

  @Override
  protected boolean requiresConfigParameter() {
    return true;
  }

  @Override
  protected boolean requiresMediaTypeAnnotation() {
    return false;
  }
}
