/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import static javax.lang.model.element.Modifier.PUBLIC;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.getXmlName;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.templating.JavaTemplateEntity;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;

import javax.lang.model.element.Modifier;

import org.apache.commons.lang3.StringUtils;

public abstract class SdkAbstractStaticTypeResolver extends JavaTemplateEntity {

  private static final String GET_SCHEMA_PATH_METHOD_NAME = "getSchemaPath";
  private static final String GET_CATEGORY_NAME_METHOD_NAME = "getCategoryName";

  protected final boolean requiresTypeResolver;
  private String className;
  private String javaPackage;
  private String schemaPath;
  private String categoryName;

  public SdkAbstractStaticTypeResolver(Path outputDir, ConnectorModel connectorModel, ConnectorOperation operation,
                                       TypeDefinition typeDefinition, SdkTypeDefinition sdkTypeDefinition) {
    super(outputDir, connectorModel);
    requiresTypeResolver = buildRequiresTypeResolver(typeDefinition, sdkTypeDefinition);
    if (requiresTypeResolver) {
      className = buildClassName(operation);
      javaPackage = buildPackage(connectorModel);
      schemaPath = sdkTypeDefinition.getSchemaPath();
      categoryName = buildCategoryName(operation);
    }
  }

  private boolean buildRequiresTypeResolver(TypeDefinition typeDefinition, SdkTypeDefinition sdkTypeDefinition) {
    return typeDefinition.getTypeSchema() != null && StringUtils.isNotBlank(sdkTypeDefinition.getSchemaPath());
  }

  protected abstract String getClassNameSuffix();

  private String buildClassName(ConnectorOperation operation) {
    return getJavaUpperCamelNameFromXml(operation.getInternalName()) + getClassNameSuffix();
  }

  private String buildCategoryName(ConnectorOperation operation) {
    return getXmlName(buildClassName(operation));
  }

  private String buildPackage(ConnectorModel connectorModel) {
    return connectorModel.getBasePackage() + ".internal.operation.metadata";
  }

  public String getClassName() {
    return className;
  }

  public boolean getRequiresTypeResolver() {
    return requiresTypeResolver;
  }

  public String getPackage() {
    return javaPackage;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (requiresTypeResolver) {
      generateTypeResolverClass();
    }
  }

  protected abstract Class<?> getSuperclass();

  private void generateTypeResolverClass() throws TemplatingException {
    TypeSpec.Builder typeResolverClassBuilder =
        TypeSpec
            .classBuilder(className)
            .addModifiers(Modifier.PUBLIC)
            .superclass(getSuperclass());

    addGetSchemaPathMethod(typeResolverClassBuilder);

    if (requiresCategoryNameOverride()) {
      addGetCategoryNameMethod(typeResolverClassBuilder);
    }

    writeClassToFile(typeResolverClassBuilder.build(), javaPackage);
  }

  private void addGetSchemaPathMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return $S", schemaPath)
        .build();

    MethodSpec createConnectionMethod = MethodSpec.methodBuilder(GET_SCHEMA_PATH_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(createConnectionMethod);
  }

  private void addGetCategoryNameMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return $S", categoryName)
        .build();

    MethodSpec createConnectionMethod = MethodSpec.methodBuilder(GET_CATEGORY_NAME_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(createConnectionMethod);
  }

  protected boolean requiresCategoryNameOverride() {
    return true;
  }

}
