/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import static org.mule.connectivity.restconnect.internal.util.FileGenerationUtils.writeSchema;

import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.type.ArrayTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.EmptyTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.XmlTypeSchema;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;

import java.io.InputStream;
import java.lang.reflect.Type;
import java.nio.file.Path;
import java.util.Date;
import java.util.List;

public class SdkTypeDefinition {

  private static final String SCHEMAS_FOLDER = "schemas";

  private final TypeDefinition typeDefinition;
  private String schemaPath;
  private String elementName;

  /***
   * Initialize type definition that wont create a type schema file.
   * @param typeDefinition The type description
   */
  public SdkTypeDefinition(TypeDefinition typeDefinition) {
    this.typeDefinition = typeDefinition;
  }

  /***
   * Initialize type definition that creates a type schema file in the resources directory.
   * This constructor should be used for the operation's input/output metadata.
   * @param sdkConnector the parent SdkConnector for this type
   * @param operation The operation for which this input/output metadata will be created
   * @param input Is this input metadata? If not it will be treated as output
   */
  public SdkTypeDefinition(SdkConnector sdkConnector, ConnectorOperation operation, boolean input) {
    this.typeDefinition = input ? operation.getInputMetadata() : operation.getOutputMetadata();

    Path resourcesDir = sdkConnector.getResourcesPath();

    if (typeRequiresSchemaGeneration()) {
      this.schemaPath = "/" + SCHEMAS_FOLDER + "/" +
          writeSchema(typeDefinition.getTypeSchema(), operation, input, resourcesDir.resolve(SCHEMAS_FOLDER),
                      sdkConnector.getTypeSchemaPaths());
      this.elementName = buildElementName(typeDefinition);
    }
  }

  private String buildElementName(TypeDefinition typeDefinition) {
    if (typeDefinition.getTypeSchema() instanceof XmlTypeSchema) {
      return ((XmlTypeSchema) typeDefinition.getTypeSchema()).getElementName();
    }

    return null;
  }

  public String getSchemaPath() {
    return schemaPath;
  }

  public String getElementName() {
    return elementName;
  }

  public Type getJavaType() {
    if (typeDefinition instanceof PrimitiveTypeDefinition) {
      return getJavaPrimitiveType((PrimitiveTypeDefinition) typeDefinition);
    } else {
      return InputStream.class;
    }
  }

  private boolean typeRequiresSchemaGeneration() {
    return getJavaType().equals(InputStream.class) && !(typeDefinition instanceof EmptyTypeDefinition);
  }

  public Type getParameterPrimitiveJavaType() {
    if (typeDefinition instanceof ArrayTypeDefinition) {
      //Should not reach here. Array Types should not depend on this method (Only for it's inner type).
      //Array's will throw this exception as they are supported for query/uri/header parameters.
      throw new IllegalArgumentException("Type is not primitive. Should handle arrays in templating.");
    }

    if (typeDefinition instanceof PrimitiveTypeDefinition && !getJavaType().equals(InputStream.class)) {
      Type javaType = getJavaType();
      if (javaType.equals(Date.class)) {
        return String.class;
      } else {
        return javaType;
      }
    } else {
      //Return String as default for not supported parameter types. (Primitive or complex)
      return String.class;
    }
  }

  public TypeName getParameterTypeName() {
    return getParameterTypeName(false);
  }

  private TypeName getParameterTypeName(boolean isInnerType) {
    if (typeDefinition instanceof ArrayTypeDefinition) {
      SdkTypeDefinition innerType =
          new SdkTypeDefinition(((ArrayTypeDefinition) typeDefinition).getInnerType());

      return ParameterizedTypeName.get(ClassName.get(List.class), innerType.getParameterTypeName(true));
    } else {
      Type parameterPrimitiveJavaType = getParameterPrimitiveJavaType();
      if (isInnerType && parameterPrimitiveJavaType.equals(boolean.class)) {
        return TypeName.get(Boolean.class);
      } else {
        return TypeName.get(parameterPrimitiveJavaType);
      }
    }
  }

  public boolean isArrayType() {
    return typeDefinition instanceof ArrayTypeDefinition;
  }

  public boolean isEnum() {
    return typeDefinition.isEnum();
  }

  public List<String> getEnumValues() {
    return typeDefinition.getEnumValues();
  }

  private Type getJavaPrimitiveType(PrimitiveTypeDefinition primitiveTypeDefinition) {
    switch (primitiveTypeDefinition.getPrimitiveType()) {
      case BOOLEAN:
        return boolean.class;
      case NUMBER:
        return Double.class;
      case INTEGER:
        return Integer.class;
      case DATE:
      case DATE_TIME:
      case DATE_ONLY:
      case DATE_TIME_ONLY:
      case TIME_ONLY:
        return Date.class;
      case FILE:
        return InputStream.class;
      case STRING:
        return String.class;
    }

    throw new IllegalArgumentException("PrimitiveTypeDefinition not supported : " + primitiveTypeDefinition.getPrimitiveType());
  }

}
