/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.ERROR;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.HTTPMethod;
import org.mule.connectivity.restconnect.internal.connectormodel.type.EmptyTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.TypeSchema;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.validation.PostValidationRule;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;

import java.util.List;

import javax.ws.rs.core.MediaType;

public class OperationMustDeclareResponseBodyRule extends PostValidationRule {

  //R003
  public OperationMustDeclareResponseBodyRule() {
    super("Operations must declare a response body",
          "GET, POST, PATCH and OPTIONS operations defined in the API spec must declare a response body and a type for that body. If they don't the connector descriptor must skip this check, define a response body for the operation, or declare it as void.",
          ERROR);
  }

  @Override
  public List<ValidationResult> postValidate(ConnectorDescriptor connectorDescriptor,
                                             APIModel apiModel,
                                             ConnectorModel connectorModel) {

    List<APIOperationModel> nullOutputAPIOperations = apiModel.getOperationsModel().stream()
        .filter(x -> httpMethodApplies(x.getHttpMethod()))
        .filter(this::hasNoTypeDefinition)
        .collect(toList());

    return nullOutputAPIOperations.stream()
        .filter(nullOutputOperation -> !validationSkipped(connectorModel, nullOutputOperation))
        .filter(nullOutputOperation -> !isVoidOperation(connectorDescriptor, nullOutputOperation))
        .filter(nullOutputOperation -> !typeDefinedInDescriptor(connectorDescriptor, nullOutputOperation))
        .map(this::getValidationError)
        .collect(toList());
  }

  private boolean isVoidOperation(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor operationDescriptor = connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && operationDescriptor.getVoidOperation() != null
        && operationDescriptor.getVoidOperation();
  }

  private boolean hasNoTypeDefinition(APIOperationModel apiOperationModel) {
    if (apiOperationModel.getOutputMetadataModel().isEmpty()) {
      return true;
    }

    return apiOperationModel.getOutputMetadataModel().stream()
        .anyMatch(OperationMustDeclareResponseBodyRule::hasNoTypeDefinition);
  }

  private static boolean hasNoTypeDefinition(APITypeModel outputType) {
    MediaType outputMediaType = outputType.getMediaType();

    if (outputMediaType != null && outputMediaType.equals(MediaType.TEXT_PLAIN_TYPE)) {
      return false;
    }

    Class<? extends TypeDefinition> outputClass = outputType.getTypeDefinitionClass();
    TypeSchema typeSchema = outputType.getAPITypeSchemaModel().getTypeSchema();

    return typeSchema == null || (outputClass == null || outputClass == EmptyTypeDefinition.class);
  }

  private boolean httpMethodApplies(HTTPMethod httpMethod) {
    return httpMethod.equals(HTTPMethod.GET)
        || httpMethod.equals(HTTPMethod.POST)
        || httpMethod.equals(HTTPMethod.PATCH)
        || httpMethod.equals(HTTPMethod.OPTIONS);
  }

  private boolean validationSkipped(ConnectorModel connectorModel, APIOperationModel apiOperationModel) {
    boolean globalSkip = connectorModel.getSkipOutputTypeValidation() != null && connectorModel.getSkipOutputTypeValidation();

    ConnectorOperation connectorOperation = connectorModel.getOperations().stream()
        .filter(x -> x.getHttpMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .findFirst().orElse(null);

    if (connectorOperation == null) {
      return false;
    }

    if (connectorOperation.getSkipOutputTypeValidation() == null) {
      return globalSkip;
    } else {
      return connectorOperation.getSkipOutputTypeValidation();
    }
  }

  private boolean typeDefinedInDescriptor(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor operationDescriptor = connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && isNotBlank(operationDescriptor.getOutputTypeSchema());
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() + " and METHOD: " + apiOperationModel.getHttpMethod().name();
    return new ValidationResult(this, location);
  }
}
