/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser;

import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.TypeSchema;

import com.fasterxml.jackson.databind.ObjectMapper;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.skyscreamer.jsonassert.JSONCompare;
import org.skyscreamer.jsonassert.JSONCompareMode;
import org.skyscreamer.jsonassert.JSONCompareResult;

public class TypeSchemaPool {

  private Map<String, TypeSchema> innerJsonPool;
  private Map<String, TypeSchema> innerSchemaPool;
  private ObjectMapper mapper = new ObjectMapper();

  public TypeSchemaPool() {
    reset();
  }

  public boolean containsTypeSchema(String rawSchema) {
    if (isJsonSchema(rawSchema)) {
      return innerJsonPool.keySet().stream().anyMatch(x -> isEqualJsonSchema(x, rawSchema));
    } else {
      return innerSchemaPool.keySet().stream().anyMatch(x -> x.equalsIgnoreCase(rawSchema));
    }
  }

  private boolean isJsonSchema(String rawSchema) {
    try {
      mapper.readTree(rawSchema);
      return true;
    } catch (IOException e) {
      return false;
    }
  }

  private boolean isEqualJsonSchema(String schemaA, String schemaB) {
    JSONCompareResult jsonCompareResult = JSONCompare.compareJSON(schemaA, schemaB, JSONCompareMode.NON_EXTENSIBLE);
    return !jsonCompareResult.failed();
  }

  public TypeSchema getTypeSchema(String rawSchema) throws ModelGenerationException {
    TypeSchema typeSchema;

    if (isJsonSchema(rawSchema)) {
      String first = innerJsonPool.keySet().stream().filter(x -> isEqualJsonSchema(x, rawSchema)).findFirst().orElse(null);
      typeSchema = innerJsonPool.get(first);
    } else {
      typeSchema = innerSchemaPool.get(rawSchema);
    }

    if (typeSchema == null) {
      throw new ModelGenerationException("This typeSchemaPool does not contain a type definition for the specified typeSchema");
    }

    return typeSchema;
  }

  public void putTypeSchema(String schema, TypeSchema typeSchema) throws ModelGenerationException {
    if (this.containsTypeSchema(schema)) {
      throw new ModelGenerationException("TypeSchema already exists in typeSchemaPool");
    }

    if (isJsonSchema(schema)) {
      innerJsonPool.put(schema, typeSchema);
    } else {
      innerSchemaPool.put(schema, typeSchema);
    }
  }

  public void reset() {
    this.innerJsonPool = new HashMap<>();
    this.innerSchemaPool = new HashMap<>();
  }
}
