/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.join;
import static org.mule.connectivity.restconnect.api.SpecFormat.RAML;
import static org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser.util.RPOperationMappingUtils.getMethods;
import org.mule.connectivity.restconnect.api.SpecFormat;
import org.mule.connectivity.restconnect.exception.InvalidSourceException;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.Protocol;
import org.mule.connectivity.restconnect.internal.connectormodel.uri.BaseUri;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser.resourceloader.RPExchangeDependencyResourceLoader;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.raml.v2.api.RamlModelBuilder;
import org.raml.v2.api.RamlModelResult;
import org.raml.v2.api.loader.CompositeResourceLoader;
import org.raml.v2.api.loader.DefaultResourceLoader;
import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.methods.Method;

public class RPAPIModel extends APIModel {

  private final Api api;

  public RPAPIModel(File apiSpec, String rootDir, SpecFormat specFormat) throws ModelGenerationException {
    if (!specFormat.equals(RAML)) {
      throw new IllegalArgumentException("RamlParser does not support " + specFormat.getName() + " format");
    }

    this.api = getAPIFromRamlFile(apiSpec, rootDir);

    this.apiName = buildApiName();
    this.description = buildDescription();
    this.protocols = buildProtocols();
    this.baseUri = buildBaseUri(protocols);
    this.operationsModel = buildOperationsModel();

  }

  private BaseUri buildBaseUri(List<Protocol> supportedProtocols) {
    String apiVersion = api.version() != null ? api.version().value() : null;
    String baseUriString = api.baseUri() != null ? api.baseUri().value() : null;

    return new BaseUri(baseUriString, apiVersion, supportedProtocols);
  }

  private String buildDescription() {
    return api.description() != null ? api.description().value() : EMPTY;
  }

  private String buildApiName() {
    return api.title() != null ? api.title().value() : EMPTY;
  }

  private List<Protocol> buildProtocols() {
    return api.protocols().stream().map(Protocol::getFromString).collect(toList());
  }

  private List<APIOperationModel> buildOperationsModel() throws ModelGenerationException {
    List<APIOperationModel> operationsModel = new ArrayList<>();

    for (Method method : getMethods(api)) {
      operationsModel.add(new RPOperationModel(api, method));
    }

    return operationsModel;
  }

  private Api getAPIFromRamlFile(File raml, String rootDir) throws InvalidSourceException {
    final CompositeResourceLoader custom =
        new CompositeResourceLoader(new DefaultResourceLoader(), new RPExchangeDependencyResourceLoader(rootDir));

    final RamlModelBuilder ramlModelBuilder = new RamlModelBuilder(custom);

    RamlModelResult ramlModelResult;
    try {
      ramlModelResult = ramlModelBuilder.buildApi(raml);
    } catch (Exception e) {
      throw new InvalidSourceException("Invalid RAML: Error in RAML Parser: " + e.getMessage() + ".", e);
    }

    if (ramlModelResult.hasErrors()) {
      String ramlErrors = join(ramlModelResult.getValidationResults(), ", ");
      throw new InvalidSourceException("Invalid RAML: " + ramlErrors + ".");
    }

    if (ramlModelResult.isVersion08()) {
      throw new InvalidSourceException("RAML 0.8 is not supported.");
    }

    if (ramlModelResult.getApiV10() == null) {
      throw new InvalidSourceException("Invalid RAML: the provided source isn't an API definition but a fragment.");
    }

    return ramlModelResult.getApiV10();
  }
}
