/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser;

import static java.util.Collections.emptyList;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.connectormodel.type.ArrayTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.EmptyTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.MultipartTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.ObjectTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.UnionTypeDefinition;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;

import org.raml.v2.api.model.v10.datamodel.ArrayTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.BooleanTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.DateTimeOnlyTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.DateTimeTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.DateTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.FileTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.JSONTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.NullTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.NumberTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.ObjectTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.StringTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.TimeOnlyTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.UnionTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.XMLTypeDeclaration;

import javax.ws.rs.core.MediaType;
import java.util.ArrayList;
import java.util.List;

public class RPTypeModel extends APITypeModel {

  private final TypeDeclaration typeDeclaration;

  public RPTypeModel(TypeDeclaration typeDeclaration, String mediaType) {
    this.typeDeclaration = typeDeclaration;

    this.example = buildExample();
    this.enumValues = buildEnumValues();
    this.mediaType = getMediaTypeForStringOrNull(mediaType);
    this.APITypeSchemaModel = new RPTypeSchemaModel(typeDeclaration, this.mediaType);
    this.typeDefinitionClass = buildTypeDefinitionClass(this.mediaType);
  }

  private Class<? extends TypeDefinition> buildTypeDefinitionClass(MediaType mediaType) {

    if (MediaType.MULTIPART_FORM_DATA_TYPE.equals(mediaType) && typeDeclaration instanceof ObjectTypeDeclaration) {
      return MultipartTypeDefinition.class;
    }

    else if (typeDeclaration instanceof ObjectTypeDeclaration
        || typeDeclaration instanceof XMLTypeDeclaration
        || typeDeclaration instanceof JSONTypeDeclaration) {
      return ObjectTypeDefinition.class;
    }

    else if (typeDeclaration instanceof ArrayTypeDeclaration) {
      return ArrayTypeDefinition.class;
    }

    else if (typeDeclaration instanceof UnionTypeDeclaration) {
      return UnionTypeDefinition.class;
    }

    else if (typeIsPrimitive(typeDeclaration)) {
      this.primitiveTypeModel = new RPPrimitiveTypeModel(typeDeclaration);
      return PrimitiveTypeDefinition.class;
    }

    else if (typeDeclaration instanceof NullTypeDeclaration) {
      return null;
    }

    else {
      // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
      return EmptyTypeDefinition.class;
    }
  }

  private List<String> buildEnumValues() {
    List<String> enumValues = null;

    if (typeDeclaration instanceof StringTypeDeclaration) {
      enumValues = ((StringTypeDeclaration) typeDeclaration).enumValues();
    }

    return enumValues;
  }

  private String buildExample() {
    return typeDeclaration.example() == null ? null : typeDeclaration.example().value();
  }

  private static boolean typeIsPrimitive(TypeDeclaration typeDeclaration) {
    return typeDeclaration instanceof BooleanTypeDeclaration ||
        typeDeclaration instanceof DateTimeOnlyTypeDeclaration ||
        typeDeclaration instanceof DateTypeDeclaration ||
        typeDeclaration instanceof DateTimeTypeDeclaration ||
        typeDeclaration instanceof FileTypeDeclaration ||
        typeDeclaration instanceof NumberTypeDeclaration ||
        typeDeclaration instanceof StringTypeDeclaration ||
        typeDeclaration instanceof TimeOnlyTypeDeclaration;
  }

  @Override
  public List<APIParameterModel> getParts() {
    if (typeDefinitionClass.equals(MultipartTypeDefinition.class)) {
      List<APIParameterModel> parts = new ArrayList<>();

      for (TypeDeclaration td : ((ObjectTypeDeclaration) typeDeclaration).properties()) {
        parts.add(new RPParameterModel(td, ParameterType.PART, false));
      }

      return parts;
    } else {
      return emptyList();
    }
  }

  @Override
  public APITypeModel getInnerType() {
    if (typeDeclaration instanceof ArrayTypeDeclaration) {
      return new RPTypeModel(((ArrayTypeDeclaration) typeDeclaration).items(), null);
    } else {
      return null;
    }
  }

  @Override
  public List<APITypeModel> getUnionTypes() {
    if (typeDeclaration instanceof UnionTypeDeclaration) {
      List<APITypeModel> innerTypes = new ArrayList<>();

      for (TypeDeclaration td : ((UnionTypeDeclaration) typeDeclaration).of()) {
        innerTypes.add(new RPTypeModel(td, null));
      }

      return innerTypes;
    } else {
      return emptyList();
    }
  }

}
