/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.INFO;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.validation.PreValidationRule;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;

import java.util.List;

public class NoDefaultOutputMediaTypeDefinedRule extends PreValidationRule {

  //R015
  public NoDefaultOutputMediaTypeDefinedRule() {
    super("No default outputMediaType was set in the Descriptor and multiple are available",
          EMPTY,
          INFO);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getOutputMetadataModel().size() > 1)
        .filter(x -> !operationHasDefaultOutputType(connectorDescriptor, x))
        .map(this::getValidationError)
        .collect(toList());
  }

  private boolean operationHasDefaultOutputType(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .anyMatch(x -> isNotBlank(x.getOutputMediaType()));
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() +
            " and METHOD: " + apiOperationModel.getHttpMethod();

    return new ValidationResult(this, location);
  }
}
