/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.descriptor.parser;

import amf.client.model.domain.DialectDomainElement;

import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import static java.lang.Boolean.parseBoolean;
import static org.mule.connectivity.restconnect.internal.descriptor.parser.DescriptorParserUtils.getSchemaContent;
import static org.mule.connectivity.restconnect.internal.descriptor.parser.DescriptorParserUtils.parseRestConnectIgnored;
import static org.mule.connectivity.restconnect.internal.descriptor.parser.DescriptorParserUtils.parseAlternativeBaseUri;

public class DescriptorOperationParser {

  private static final String API_CONTRACT_SUPPORTED_OPERATION = "http://a.ml/vocabularies/apiContract#supportedOperation";
  private static final String API_CONTRACT_METHOD = "http://a.ml/vocabularies/apiContract#method";

  private static final String REST_CONNECT_OPERATION_NAME = "http://a.ml/vocabularies/restConnect#operationName";
  private static final String REST_CONNECT_OPERATION_DESCRIPTION = "http://a.ml/vocabularies/restConnect#operationDescription";
  private static final String REST_CONNECT_DEFAULT_INPUT_MEDIA_TYPE = "http://a.ml/vocabularies/restConnect#inputMediaType";
  private static final String REST_CONNECT_DEFAULT_OUTPUT_MEDIA_TYPE = "http://a.ml/vocabularies/restConnect#outputMediaType";
  private static final String REST_CONNECT_INPUT_TYPE = "http://a.ml/vocabularies/restConnect#inputType";
  private static final String REST_CONNECT_OUTPUT_TYPE = "http://a.ml/vocabularies/restConnect#outputType";
  private static final String REST_CONNECT_PAGINATION = "http://a.ml/vocabularies/restConnect#paginationReference";
  private static final String REST_CONNECT_VOID_OPERATION = "http://a.ml/vocabularies/restConnect#voidOperation";

  private DescriptorRequestParser descriptorRequestParser = new DescriptorRequestParser();
  private DescriptorEncodesParser descriptorEncodesParser = new DescriptorEncodesParser();


  public List<OperationDescriptor> parseOperations(DialectDomainElement endpointElement, Path descriptorFilePath)
      throws ModelGenerationException {

    List<OperationDescriptor> operationDescriptors = new ArrayList<>();

    for (DialectDomainElement x : getSupportedOperations(endpointElement)) {
      operationDescriptors.add(parseOperation(x, descriptorFilePath));
    }

    return operationDescriptors;
  }

  private OperationDescriptor parseOperation(DialectDomainElement operationElement, Path descriptorFilePath)
      throws ModelGenerationException {
    Path descriptorDirectory = descriptorFilePath.getParent();

    return new OperationDescriptor(
                                   getMethod(operationElement),
                                   getName(operationElement),
                                   getDescription(operationElement),
                                   getDefaultInputMediaType(operationElement),
                                   getDefaultOutputMediaType(operationElement),
                                   descriptorRequestParser.parseRequest(operationElement, descriptorDirectory),
                                   parseRestConnectIgnored(operationElement),
                                   parseAlternativeBaseUri(operationElement),
                                   getPagination(operationElement),
                                   parseSkipOutputTypeValidation(operationElement),
                                   parseVoidOperation(operationElement),
                                   getInputTypeSchema(operationElement, descriptorDirectory),
                                   getOutputTypeSchema(operationElement, descriptorDirectory));
  }

  private List<DialectDomainElement> getSupportedOperations(DialectDomainElement endpointElement) {
    return endpointElement.getObjectPropertyUri(API_CONTRACT_SUPPORTED_OPERATION);
  }

  private Boolean parseSkipOutputTypeValidation(DialectDomainElement operationElement) {
    return descriptorEncodesParser.parseSkipOutputTypeValidation(operationElement);
  }

  private Boolean parseVoidOperation(DialectDomainElement operationElement) {
    List<Object> voidOperations = operationElement.getScalarByPropertyUri(REST_CONNECT_VOID_OPERATION);
    return voidOperations.isEmpty() ? null : parseBoolean(voidOperations.get(0).toString());
  }

  public String getMethod(DialectDomainElement operationElement) {
    return operationElement.getScalarByPropertyUri(API_CONTRACT_METHOD).get(0).toString();
  }

  private String getDescription(DialectDomainElement operationElement) {
    List<Object> apiOperationDescription = operationElement.getScalarByPropertyUri(REST_CONNECT_OPERATION_DESCRIPTION);
    return apiOperationDescription.isEmpty() ? null : apiOperationDescription.get(0).toString();
  }

  private String getName(DialectDomainElement operationElement) {
    List<Object> operationNamesMetadata = operationElement.getScalarByPropertyUri(REST_CONNECT_OPERATION_NAME);
    return operationNamesMetadata.isEmpty() ? null : operationNamesMetadata.get(0).toString();
  }

  private String getDefaultInputMediaType(DialectDomainElement operationElement) {
    List<Object> operationNamesMetadata = operationElement.getScalarByPropertyUri(REST_CONNECT_DEFAULT_INPUT_MEDIA_TYPE);
    return operationNamesMetadata.isEmpty() ? null : operationNamesMetadata.get(0).toString();
  }

  private String getInputTypeSchema(DialectDomainElement operationElement, Path descriptorDirectory)
      throws ModelGenerationException {
    List<Object> operationNamesMetadata = operationElement.getScalarByPropertyUri(REST_CONNECT_INPUT_TYPE);
    String inputSchemaPathString = operationNamesMetadata.isEmpty() ? null : operationNamesMetadata.get(0).toString();

    return getSchemaContent(descriptorDirectory, inputSchemaPathString);
  }

  private String getOutputTypeSchema(DialectDomainElement operationElement, Path descriptorDirectory)
      throws ModelGenerationException {
    List<Object> operationNamesMetadata = operationElement.getScalarByPropertyUri(REST_CONNECT_OUTPUT_TYPE);
    String inputSchemaPathString = operationNamesMetadata.isEmpty() ? null : operationNamesMetadata.get(0).toString();

    return getSchemaContent(descriptorDirectory, inputSchemaPathString);
  }

  private String getDefaultOutputMediaType(DialectDomainElement operationElement) {
    List<Object> operationNamesMetadata = operationElement.getScalarByPropertyUri(REST_CONNECT_DEFAULT_OUTPUT_MEDIA_TYPE);
    return operationNamesMetadata.isEmpty() ? null : operationNamesMetadata.get(0).toString();
  }

  private String getPagination(DialectDomainElement operationElement) {
    List<Object> paginationName = operationElement.getScalarByPropertyUri(REST_CONNECT_PAGINATION);
    return paginationName.isEmpty() ? null : paginationName.get(0).toString();
  }

}
