/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.TypeSpec;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.TypeSchema;
import org.mule.connectivity.restconnect.internal.templating.JavaTemplateEntity;
import org.mule.connectivity.restconnect.internal.util.JavaUtils;
import org.mule.connectors.restconnect.commons.api.error.RestConnectError;
import org.mule.runtime.api.meta.Category;
import org.mule.runtime.extension.api.annotation.Configurations;
import org.mule.runtime.extension.api.annotation.Extension;
import org.mule.runtime.extension.api.annotation.dsl.xml.Xml;
import org.mule.runtime.extension.api.annotation.error.ErrorTypes;
import org.mule.runtime.extension.api.annotation.license.RequiresEnterpriseLicense;

import javax.lang.model.element.Modifier;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import static com.google.common.base.CaseFormat.LOWER_HYPHEN;
import static com.google.common.base.CaseFormat.UPPER_CAMEL;
import static java.util.stream.Collectors.toList;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static org.mule.connectivity.restconnect.internal.webapi.util.ParserUtils.splitCaps;

public class SdkConnector extends JavaTemplateEntity {

  private final List<SdkConfig> configs;
  private final Map<TypeSchema, String> typeSchemaPaths = new HashMap<>();
  private final Map<List<String>, String> enumValuesName = new HashMap<>();

  public SdkConnector(Path outputDir, ConnectorModel connectorModel) {
    super(outputDir, connectorModel);

    this.configs = Stream.of(new SdkConfig(outputDir, connectorModel, this)).collect(toList());
  }

  public String getJavaName() {
    return getJavaUpperCamelNameFromXml(connectorModel.getConnectorXmlName());
  }

  private Category getCategory() {
    return connectorModel.getCategory() != null ? Category.valueOf(connectorModel.getCategory().getCategoryName().toUpperCase())
        : null;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConnectorClass();

    for (SdkConfig config : configs) {
      config.applyTemplates();
    }
  }

  private void generateConnectorClass() throws TemplatingException {

    TypeSpec.Builder connectorClassBuilder =
        TypeSpec
            .classBuilder(JavaUtils.getJavaUpperCamelNameFromXml(connectorModel.getConnectorXmlName()))
            .addModifiers(Modifier.PUBLIC)
            .addAnnotation(getExtensionAnnotation())
            .addAnnotation(getXmlAnnotation())
            .addAnnotation(getConfigurationsAnnotation())
            .addAnnotation(getErrorTypesAnnotation());

    if (getCategory() != null) {
      connectorClassBuilder.addAnnotation(getLicenseAnnotation());
    }

    writeClassToFile(connectorClassBuilder.build(), getPackage());
  }

  private String getPackage() {
    return connectorModel.getBasePackage() + ".internal";
  }

  private AnnotationSpec getExtensionAnnotation() {
    String extensionName = LOWER_HYPHEN.to(UPPER_CAMEL, connectorModel.getConnectorXmlName());
    extensionName = splitCaps(extensionName, " ");

    AnnotationSpec.Builder extensionAnnotationBuilder = AnnotationSpec.builder(Extension.class)
        .addMember(NAME_MEMBER, "$S", extensionName);

    if (getCategory() != null) {
      extensionAnnotationBuilder.addMember("category", "$T.$L", Category.class, getCategory());
    }

    return extensionAnnotationBuilder.build();
  }

  private AnnotationSpec getLicenseAnnotation() {
    AnnotationSpec.Builder connectorAnnotationBuilder = AnnotationSpec.builder(RequiresEnterpriseLicense.class)
        .addMember("allowEvaluationLicense", "$L", true);

    return connectorAnnotationBuilder.build();
  }

  private AnnotationSpec getXmlAnnotation() {
    AnnotationSpec.Builder connectorAnnotationBuilder = AnnotationSpec.builder(Xml.class)
        .addMember("prefix", "$S", connectorModel.getExtensionXml());

    return connectorAnnotationBuilder.build();
  }

  private AnnotationSpec getConfigurationsAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < this.configs.size(); i++) {
      SdkConfig config = this.configs.get(i);
      codeBlock.add("$T.class", ClassName.get(config.getPackage(), config.getJavaClassName()));
      if (i < this.configs.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder configurationsAnnotationBuilder =
        AnnotationSpec
            .builder(Configurations.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return configurationsAnnotationBuilder.build();
  }

  private AnnotationSpec getErrorTypesAnnotation() {
    return AnnotationSpec
        .builder(ErrorTypes.class)
        .addMember(VALUE_MEMBER, "$T.class", RestConnectError.class)
        .build();
  }

  public Map<TypeSchema, String> getTypeSchemaPaths() {
    return typeSchemaPaths;
  }

  public Map<List<String>, String> getEnumValuesName() {
    return enumValuesName;
  }
}
