/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import static org.mule.connectivity.restconnect.internal.templating.sdk.SdkContent.SdkContentKind.PART;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.type.MultipartTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.annotation.Nullable;

public class SdkMultipartOperation extends SdkOperation {

  private final List<SdkMultipartPart> sdkParts;

  private class SdkMultipartPart {

    private final PartParameter partParameter;
    private final SdkContent partContent;
    private final SdkParameter partFilename;

    SdkMultipartPart(PartParameter partParameter, SdkContent partContent, @Nullable SdkParameter partFilename) {
      this.partParameter = partParameter;
      this.partContent = partContent;
      this.partFilename = partFilename;
    }

    public PartParameter getPartParameter() {
      return partParameter;
    }

    public SdkContent getPartContent() {
      return partContent;
    }

    public SdkParameter getPartFilename() {
      return partFilename;
    }
  }

  public SdkMultipartOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                               ConnectorOperation operation) {
    super(outputDir, connectorModel, sdkConnector, operation);

    MultipartTypeDefinition multipartContent = (MultipartTypeDefinition) operation.getInputMetadata();

    this.sdkParts = buildSdkMultipartParts(outputDir, connectorModel, sdkConnector, operation, multipartContent);
  }

  protected List<SdkMultipartPart> buildSdkMultipartParts(Path outputDir, ConnectorModel connectorModel,
                                                          SdkConnector sdkConnector,
                                                          ConnectorOperation operation,
                                                          MultipartTypeDefinition multipartContent) {
    List<SdkMultipartPart> parts = new ArrayList<>();

    boolean primary = true;
    for (PartParameter partParameter : multipartContent.getParts()) {

      SdkContent partContent =
          new SdkContent(
                         outputDir,
                         connectorModel,
                         sdkConnector,
                         operation,
                         partParameter.getTypeDefinition(),
                         PART,
                         partParameter.getExternalName(),
                         primary);

      SdkParameter filenameParameter = partParameter.isFilePart() ? new SdkParameter(outputDir,
                                                                                     connectorModel,
                                                                                     sdkConnector,
                                                                                     getJavaClassName(),
                                                                                     getFilenameParameter(partParameter))
          : null;

      SdkMultipartPart sdkMultipartPart = new SdkMultipartPart(
                                                               partParameter,
                                                               partContent,
                                                               filenameParameter);

      parts.add(sdkMultipartPart);
      primary = false;
    }

    return parts;
  }

  private Parameter getFilenameParameter(PartParameter partParameter) {
    return new Parameter(partParameter.getDisplayName() + " Filename",
                         partParameter.getExternalName() + "-part-filename",
                         ParameterType.PART,
                         TypeDefinition.simpleStringType(),
                         "The name of the file being sent in the '"
                             + partParameter.getDisplayName()
                             + "' part. (just the name, do not include path).",
                         true,
                         null,
                         false,
                         new LinkedList<>());
  }

  @Override
  protected void addSetBodyMethod(CodeBlock.Builder methodBody) {
    methodBody.addStatement("setBody(builder, body)");
  }

  @Override
  public CodeBlock generateOperationMethodBody() {
    CodeBlock consumerBody = generateOperationMethodBody("multipartCallback");

    return CodeBlock.builder()
        .add("withMultipart(builder -> $L , callback, (body, multipartCallback) -> { $L });",
             generateMultipartBuilderCodeBlock(),
             consumerBody)
        .build();
  }

  protected CodeBlock generateMultipartBuilderCodeBlock() {
    CodeBlock.Builder multipartBuilderCodeBlockBuilder = CodeBlock.builder().add("builder");

    if (!sdkParts.isEmpty()) {
      for (SdkMultipartPart sdkPart : sdkParts) {
        if (sdkPart.getPartParameter().isFilePart()) {
          multipartBuilderCodeBlockBuilder
              .add(".addFilePart($S, $L, $L)",
                   sdkPart.getPartParameter().getExternalName(),
                   sdkPart.getPartFilename().getStringValueGetter(),
                   sdkPart.getPartContent().getContentParameterJavaName());
        } else {
          multipartBuilderCodeBlockBuilder
              .add(".addPart($S, $L)",
                   sdkPart.getPartParameter().getExternalName(),
                   sdkPart.getPartContent().getContentParameterJavaName());
        }
      }

      multipartBuilderCodeBlockBuilder
          .add(".setBoundary(\"__rc2_34b212\")");
    }


    return multipartBuilderCodeBlockBuilder.build();
  }

  @Override
  protected void addContentParameters(CodeBlock.Builder javaDoc, MethodSpec.Builder methodBuilder) {
    for (SdkMultipartPart sdkPart : sdkParts) {

      methodBuilder.addParameter(sdkPart.getPartContent().generateContentParameter());
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION,
                  sdkPart.getPartContent().getContentParameterJavaName(),
                  "The content of the '" + sdkPart.getPartParameter().getDisplayName() + "' part.");

      if (sdkPart.getPartFilename() != null) {
        methodBuilder.addParameter(sdkPart.getPartFilename().generateParameterParameter().build());
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION,
                    sdkPart.getPartFilename().getJavaName(),
                    "The filename of the '" + sdkPart.getPartParameter().getDisplayName()
                        + "' part (just the name, do not include path).");
      }
    }
  }

  @Override
  protected SdkContent buildContent(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                    ConnectorOperation operation) {
    return null;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    for (SdkMultipartPart sdkPart : sdkParts) {
      sdkPart.getPartContent().applyTemplates();
      if (sdkPart.getPartFilename() != null) {
        sdkPart.getPartFilename().applyTemplates();
      }
    }

    super.applyTemplates();
  }

}
