/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.EndPointDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.PartDescriptor;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.TypeSchemaPool;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.ws.rs.core.MediaType;

public class ConnectorOperationBuilder {

  private final ConnectorTypeDefinitionBuilder typeDefinitionBuilder;
  private final ConnectorParameterBuilder parameterBuilder;
  private final ConnectorSecuritySchemeBuilder securitySchemeBuilder;

  public ConnectorOperationBuilder(TypeSchemaPool typeSchemaPool) {
    this.typeDefinitionBuilder = new ConnectorTypeDefinitionBuilder(typeSchemaPool);
    this.parameterBuilder = new ConnectorParameterBuilder(typeDefinitionBuilder);
    this.securitySchemeBuilder = new ConnectorSecuritySchemeBuilder(typeDefinitionBuilder);
  }

  public List<ConnectorOperation> buildOperations(APIModel apiModel, ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {

    List<ConnectorOperation> operations = new ArrayList<>();
    for (APIOperationModel op : apiModel.getOperationsModel()) {
      ConnectorOperation connectorOperation = buildOperation(op,
                                                             getEndpointDescriptor(op, connectorDescriptor),
                                                             getOperationDescriptor(op, connectorDescriptor),
                                                             connectorDescriptor);
      if (connectorOperation != null) {
        operations.add(connectorOperation);
      }
    }

    return operations;
  }

  private static EndPointDescriptor getEndpointDescriptor(APIOperationModel operationModel,
                                                          ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(operationModel.getPath()))
        .findFirst().orElse(null);
  }

  private static OperationDescriptor getOperationDescriptor(APIOperationModel operationModel,
                                                            ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(operationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(operationModel.getHttpMethod().name()))
        .findFirst().orElse(null);
  }

  public ConnectorOperation buildOperation(APIOperationModel apiOperationModel, EndPointDescriptor endPointDescriptor,
                                           OperationDescriptor operationDescriptor, ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {

    if (endPointDescriptor != null && endPointDescriptor.isIgnored()) {
      return null;
    }

    if (operationDescriptor != null && operationDescriptor.isIgnored()) {
      return null;
    }

    List<String> collisionNames = new LinkedList<>();
    List<Parameter> uriParameters =
        parameterBuilder.buildParameterList(apiOperationModel.getUriParamsModel(), operationDescriptor, collisionNames);
    collisionNames.addAll(uriParameters.stream().map(Parameter::getInternalName).collect(toList()));

    List<Parameter> queryParameters =
        parameterBuilder.buildParameterList(apiOperationModel.getQueryParamsModel(), operationDescriptor, collisionNames);
    collisionNames.addAll(queryParameters.stream().map(Parameter::getInternalName).collect(toList()));

    List<Parameter> headers =
        parameterBuilder.buildParameterList(apiOperationModel.getHeadersModel(), operationDescriptor, collisionNames);

    return new ConnectorOperation(
                                  buildOperationName(apiOperationModel, operationDescriptor),
                                  buildOperationDescription(apiOperationModel, operationDescriptor),
                                  apiOperationModel.getPath(),
                                  apiOperationModel.getHttpMethod(),
                                  uriParameters,
                                  queryParameters,
                                  headers,
                                  buildInputTypeMetadata(apiOperationModel, operationDescriptor, connectorDescriptor),
                                  buildOutputTypeMetadata(apiOperationModel, operationDescriptor, connectorDescriptor),
                                  securitySchemeBuilder.buildSecuritySchemes(apiOperationModel.getSecuritySchemesModel(),
                                                                             connectorDescriptor.getSecurity()),
                                  buildAlternativeBaseUri(endPointDescriptor, operationDescriptor),
                                  buildPagination(operationDescriptor),
                                  buildSkipOutputTypeValidation(operationDescriptor),
                                  buildVoidOperation(operationDescriptor));
  }

  private static Boolean buildSkipOutputTypeValidation(OperationDescriptor operationDescriptor) {
    return operationDescriptor != null ? operationDescriptor.getSkipOutputTypeValidation() : null;
  }

  private static Boolean buildVoidOperation(OperationDescriptor operationDescriptor) {
    return operationDescriptor != null ? operationDescriptor.getVoidOperation() : null;
  }

  private static String buildOperationName(APIOperationModel operationModel, OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getName())) {
      return operationDescriptor.getName();
    } else {
      return operationModel.getName();
    }
  }

  private static String buildOperationDescription(APIOperationModel operationModel, OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getDescription())) {
      return operationDescriptor.getDescription();
    } else {
      return operationModel.getDescription();
    }
  }

  private static String buildAlternativeBaseUri(EndPointDescriptor endPointDescriptor, OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getBaseUri())) {
      return operationDescriptor.getBaseUri();
    } else if (endPointDescriptor != null && isNotBlank(endPointDescriptor.getBaseUri())) {
      return endPointDescriptor.getBaseUri();
    }
    return null;
  }

  private static String buildPagination(OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getPagination())) {
      return operationDescriptor.getPagination();
    }

    return null;
  }

  private TypeDefinition buildInputTypeMetadata(APIOperationModel apiOperationModel,
                                                OperationDescriptor operationDescriptor,
                                                ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {

    String descriptorInputMediaType = EMPTY;
    String descriptorInputTypeSchema = EMPTY;
    List<PartDescriptor> partDescriptors = null;
    if (operationDescriptor != null) {
      descriptorInputMediaType = operationDescriptor.getInputMediaType();
      descriptorInputTypeSchema = operationDescriptor.getInputTypeSchema();
      partDescriptors = operationDescriptor.getExpects() != null ? operationDescriptor.getExpects().getPart() : null;
    }

    return buildIOTypeMetadata(apiOperationModel.getInputMetadataModel(),
                               connectorDescriptor.getDefaultInputMediaType(),
                               descriptorInputMediaType,
                               descriptorInputTypeSchema,
                               partDescriptors);
  }

  private TypeDefinition buildOutputTypeMetadata(APIOperationModel apiOperationModel,
                                                 OperationDescriptor operationDescriptor,
                                                 ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {
    String descriptorOutputMediaType = EMPTY;
    String descriptorOutputTypeSchema = EMPTY;

    if (operationDescriptor != null) {
      descriptorOutputMediaType = operationDescriptor.getOutputMediaType();
      descriptorOutputTypeSchema = operationDescriptor.getOutputTypeSchema();
    }

    return buildIOTypeMetadata(apiOperationModel.getOutputMetadataModel(),
                               connectorDescriptor.getDefaultOutputMediaType(),
                               descriptorOutputMediaType,
                               descriptorOutputTypeSchema,
                               null /*There is no support for output multipart descriptors*/);
  }

  private TypeDefinition buildIOTypeMetadata(List<APITypeModel> metadataModels,
                                             String descriptorGlobalMediaType,
                                             String descriptorMediaType,
                                             String descriptorTypeSchema,
                                             List<PartDescriptor> partDescriptors)
      throws ModelGenerationException {
    MediaType inputMediaType = null;

    if (isNotBlank(descriptorMediaType)) {
      inputMediaType = MediaType.valueOf(descriptorMediaType);
    } else if (isNotBlank(descriptorGlobalMediaType)) {
      inputMediaType = MediaType.valueOf(descriptorGlobalMediaType);
    }

    if (isNotBlank(descriptorTypeSchema)) {
      if (inputMediaType == null) {
        APITypeModel defaultApiTypeModel = getDefaultApiTypeModel(metadataModels);
        inputMediaType = defaultApiTypeModel.getMediaType();
      }

      return typeDefinitionBuilder.buildTypeDefinition(descriptorTypeSchema, inputMediaType);
    }

    return buildTypeMetadata(metadataModels, inputMediaType, partDescriptors);
  }

  private TypeDefinition buildTypeMetadata(List<APITypeModel> metadataModels, MediaType mediaType,
                                           List<PartDescriptor> partDescriptors)
      throws ModelGenerationException {

    if (!metadataModels.isEmpty()) {
      //API defines metadata media type ->
      //-- If descriptor defines one it must exist in the API spec and it will be used to generate the connector
      //-- If descriptor does not define a media type, use the default one from the API definitions
      //API does not define media type ->
      //-- Descriptor must define a media type, and it will be used for the first metadata defined in the API spec

      boolean metadataModelsDefineMediaType = metadataModels.stream().anyMatch(x -> x.getMediaType() != null);

      if (metadataModelsDefineMediaType) {
        APITypeModel apiTypeModel = getDefaultApiTypeModel(metadataModels);

        if (mediaType != null) {
          apiTypeModel = getApiTypeModelByMediaType(metadataModels, mediaType);
          if (apiTypeModel == null) {
            throw new ModelGenerationException("Could not get body with media type from api spec: " + mediaType);
          }
        }

        return typeDefinitionBuilder.buildTypeDefinition(apiTypeModel, partDescriptors);
      } else {
        APITypeModel apiTypeModel = getDefaultApiTypeModel(metadataModels);
        return typeDefinitionBuilder.buildTypeDefinition(apiTypeModel, partDescriptors, mediaType);
      }
    }

    return null;
  }

  private static APITypeModel getDefaultApiTypeModel(List<APITypeModel> metadataModels) {
    APITypeModel apiTypeModel = getApiTypeModelByMediaType(metadataModels, MediaType.APPLICATION_JSON_TYPE);

    if (apiTypeModel == null) {
      apiTypeModel = getApiTypeModelByMediaType(metadataModels, MediaType.APPLICATION_XML_TYPE);
    }

    return apiTypeModel == null ? metadataModels.get(0) : apiTypeModel;
  }

  private static APITypeModel getApiTypeModelByMediaType(List<APITypeModel> metadataModels, MediaType mediaType) {
    return metadataModels.stream()
        .filter(x -> x.getMediaType() != null && x.getMediaType().equals(mediaType))
        .findFirst().orElse(null);
  }
}
