/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_XML_TYPE;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.BOOLEAN;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.DATE;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.DATE_ONLY;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.DATE_TIME;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.DATE_TIME_ONLY;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.FILE;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.INTEGER;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.NUMBER;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.STRING;
import static org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition.PrimitiveType.TIME_ONLY;
import static org.mule.connectivity.restconnect.internal.util.FileGenerationUtils.SchemaNameType.OUTPUT;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.type.PrimitiveTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.CustomTypeSchema;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.JsonTypeSchema;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.TypeSchema;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.XmlTypeSchema;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.BooleanOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.DateOnlyOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.DateTimeOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.FileOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.FromJsonOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.IntegerOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.JsonOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.NumberOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.StringOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.TimeOnlyOutputTypeResolver;
import org.mule.connectors.restconnect.commons.api.datasense.resolver.output.XmlOutputTypeResolver;
import org.mule.runtime.extension.api.annotation.metadata.OutputResolver;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;

import java.nio.file.Path;

public class SdkOutputTypeResolver extends SdkAbstractStaticTypeResolver {

  private String qName;

  public SdkOutputTypeResolver(Path outputDir,
                               ConnectorModel connectorModel,
                               SdkConnector sdkConnector,
                               ConnectorOperation operation,
                               TypeDefinition typeDefinition)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, operation, typeDefinition, OUTPUT, EMPTY);
  }

  protected Class<?> buildSuperclass(TypeDefinition typeDefinition) throws TemplatingException {
    TypeSchema typeSchema = typeDefinition.getTypeSchema();

    if (typeDefinition instanceof PrimitiveTypeDefinition) {
      return getPrimitiveTypeResolverClass((PrimitiveTypeDefinition) typeDefinition);
    } else if (typeSchema instanceof XmlTypeSchema || typeDefinition.getMediaType().isCompatible(APPLICATION_XML_TYPE)) {
      qName = ((XmlTypeSchema) typeDefinition.getTypeSchema()).getElementName();
      return XmlOutputTypeResolver.class;
    } else if (typeSchema instanceof JsonTypeSchema || typeDefinition.getMediaType().isCompatible(APPLICATION_JSON_TYPE)) {
      return JsonOutputTypeResolver.class;
    } else if (typeSchema instanceof CustomTypeSchema) {
      return FromJsonOutputTypeResolver.class;
    }

    return StringOutputTypeResolver.class;
  }

  protected Class<?> getPrimitiveTypeResolverClass(PrimitiveTypeDefinition primitiveTypeDefinition) throws TemplatingException {
    PrimitiveTypeDefinition.PrimitiveType primitiveType = primitiveTypeDefinition.getPrimitiveType();

    if (primitiveType.equals(NUMBER)) {
      return NumberOutputTypeResolver.class;
    }

    if (primitiveType.equals(INTEGER)) {
      return IntegerOutputTypeResolver.class;
    }

    if (primitiveType.equals(STRING)) {
      return StringOutputTypeResolver.class;
    }

    if (primitiveType.equals(BOOLEAN)) {
      return BooleanOutputTypeResolver.class;
    }

    if (primitiveType.equals(DATE_TIME)
        || primitiveType.equals(DATE_TIME_ONLY)) {
      return DateTimeOutputTypeResolver.class;
    }

    if (primitiveType.equals(TIME_ONLY)) {
      return TimeOnlyOutputTypeResolver.class;
    }

    if (primitiveType.equals(DATE_ONLY)
        || primitiveType.equals(DATE)) {
      return DateOnlyOutputTypeResolver.class;
    }

    if (primitiveType.equals(FILE)) {
      return FileOutputTypeResolver.class;
    }

    throw new TemplatingException("Primitive type not supported: " + primitiveType.name());
  }

  public AnnotationSpec getOutputTypeResolverAnnotation() {
    return AnnotationSpec
        .builder(OutputResolver.class)
        .addMember("output", "$T.class", ClassName.get(getPackage(), getClassName()))
        .build();
  }

  @Override
  protected String getClassNameSuffix() {
    return "OutputTypeResolver";
  }

  @Override
  protected boolean requiresQNameMethod() {
    return superclass.equals(XmlOutputTypeResolver.class);
  }

  @Override
  protected boolean requiresFormatMethod() {
    return superclass.equals(FromJsonOutputTypeResolver.class)
        || superclass.equals(FileOutputTypeResolver.class);
  }

  @Override
  protected String getQName() {
    return qName;
  }

  @Override
  protected boolean requiresToWriteSchema() {
    return superclass.equals(FromJsonOutputTypeResolver.class)
        || superclass.equals(JsonOutputTypeResolver.class)
        || superclass.equals(XmlOutputTypeResolver.class);
  }

}
