/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.parser.amf;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import org.mule.connectivity.restconnect.api.SpecFormat;
import org.mule.connectivity.restconnect.exception.InvalidSourceException;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.Protocol;
import org.mule.connectivity.restconnect.internal.connectormodel.uri.BaseUri;
import org.mule.connectivity.restconnect.internal.util.AMFWrapper;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.amf.resourceloader.AMFExchangeDependencyResourceLoader;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import amf.ProfileNames;
import amf.client.AMF;
import amf.client.environment.DefaultEnvironment;
import amf.client.environment.Environment;
import amf.client.model.document.BaseUnit;
import amf.client.model.document.Document;
import amf.client.model.domain.EndPoint;
import amf.client.model.domain.Server;
import amf.client.model.domain.WebApi;
import amf.client.parse.Oas20Parser;
import amf.client.parse.Oas20YamlParser;
import amf.client.parse.Parser;
import amf.client.parse.Raml10Parser;
import amf.client.validate.ValidationReport;

public class AMFAPIModel extends APIModel {

  private final SpecFormat specFormat;
  private final WebApi webApi;

  public AMFAPIModel(File apiSpec, String rootDir, SpecFormat specFormat, boolean skipValidations)
      throws ModelGenerationException {
    AMFWrapper.initialize();

    this.specFormat = specFormat;

    this.webApi = getWebApi(getParser(specFormat, rootDir), apiSpec.toPath(), skipValidations);

    this.apiName = buildApiName();
    this.description = buildDescription();
    this.protocols = buildProtocols();
    this.baseUri = buildBaseUri(protocols);
    this.operationsModel = buildOperationsModel();
  }

  private String buildApiName() {
    return webApi.name().value();
  }

  private String buildDescription() {
    return webApi.description().nonEmpty() ? webApi.description().value() : EMPTY;
  }

  private BaseUri buildBaseUri(List<Protocol> supportedProtocols) {
    String apiVersion = webApi.version() != null ? webApi.version().value() : null;

    String baseUri = null;
    Server server = webApi.servers().stream().filter(x -> x.url().nonEmpty()).findFirst().orElse(null);
    if (server != null) {
      baseUri = server.url().value();
    }

    return new BaseUri(baseUri, apiVersion, supportedProtocols);
  }

  private List<Protocol> buildProtocols() {
    return webApi.schemes().stream().map(x -> Protocol.getFromString(x.value())).collect(toList());
  }

  private List<APIOperationModel> buildOperationsModel() throws ModelGenerationException {
    List<APIOperationModel> operationsModel = new ArrayList<>();

    for (EndPoint endPoint : webApi.endPoints()) {
      for (amf.client.model.domain.Operation operation : endPoint.operations()) {
        operationsModel.add(new AMFOperationModel(endPoint, operation));
      }
    }

    return operationsModel;
  }

  private Parser getParser(SpecFormat format, String rootDir) {
    final Environment env = DefaultEnvironment.apply().add(new AMFExchangeDependencyResourceLoader(rootDir));

    if (format.equals(SpecFormat.RAML)) {
      return new Raml10Parser(env);
    } else if (format.equals(SpecFormat.JSON_OAS)) {
      return new Oas20Parser(env);
    } else if (format.equals(SpecFormat.YAML_OAS)) {
      return new Oas20YamlParser(env);
    }

    throw new IllegalArgumentException("Spec format " + format.getName() + " not supported");
  }

  private WebApi getWebApi(Parser parser, Path path, boolean skipValidations) throws InvalidSourceException {
    if (isRaml08(path.toAbsolutePath())) {
      throw new InvalidSourceException("RAML 0.8 is not supported.");
    }

    Document ramlDocument;
    try {
      ramlDocument = parseFile(parser, "file://" + path.toAbsolutePath().toString());
      if (!skipValidations) {
        validate(ramlDocument, specFormat);
      }

      ramlDocument = resolve(ramlDocument, specFormat);
    } catch (Exception e) {
      throw new InvalidSourceException("Invalid Spec: Error in AMF: " + e.getMessage() + ".", e);
    }

    return getWebApi(ramlDocument);
  }

  private Document resolve(Document ramlDocument, SpecFormat specFormat) {
    if (specFormat.equals(SpecFormat.RAML)) {
      return (Document) AMF.resolveRaml10(ramlDocument);
    } else if (specFormat.equals(SpecFormat.JSON_OAS) || specFormat.equals(SpecFormat.YAML_OAS)) {
      return (Document) AMF.resolveOas20(ramlDocument);
    }
    throw new IllegalArgumentException("Unrecognised spec format");
  }

  private void validate(Document ramlDocument, SpecFormat specFormat)
      throws ExecutionException, InterruptedException, InvalidSourceException {
    ValidationReport report;

    if (specFormat.equals(SpecFormat.RAML)) {
      report = AMF.validate(ramlDocument, ProfileNames.RAML(), ProfileNames.AMF().messageStyle()).get();
      if (!report.conforms()) {
        throw new InvalidSourceException(report.toString());
      }
    } else if (specFormat.equals(SpecFormat.JSON_OAS) || specFormat.equals(SpecFormat.YAML_OAS)) {
      report = AMF.validate(ramlDocument, ProfileNames.OAS(), ProfileNames.AMF().messageStyle()).get();
      if (!report.conforms()) {
        throw new InvalidSourceException(report.toString());
      }
    } else {
      throw new IllegalArgumentException("Unrecognised spec format");
    }
  }

  private WebApi getWebApi(Document document) {
    return (WebApi) document.encodes();
  }

  private Document parseFile(Parser parser, String url) {
    return handleFuture(parser.parseFileAsync(url));
  }

  private Document handleFuture(CompletableFuture<BaseUnit> f) {
    try {
      return (Document) f.get();
    } catch (ExecutionException e) {
      throw new RuntimeException("An error occurred while parsing the api. Message: " + e.getMessage(), e);
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
      throw new RuntimeException("An error occurred while parsing the api. Message: " + e.getMessage(), e);
    }
  }

  private boolean isRaml08(Path specPath) throws InvalidSourceException {
    String content;

    try {
      content = new String(java.nio.file.Files.readAllBytes(specPath), StandardCharsets.UTF_8);
    } catch (IOException e) {
      throw new InvalidSourceException("Could not load raml file content", e);
    }

    return content.startsWith("#%RAML 0.8");
  }

}
