/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.util.FileGenerationUtils.writeSchema;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.getXmlName;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.type.EmptyTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.templating.JavaTemplateEntity;
import org.mule.connectivity.restconnect.internal.util.FileGenerationUtils.SchemaNameType;
import org.mule.metadata.api.model.MetadataFormat;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;

import javax.lang.model.element.Modifier;

public abstract class SdkAbstractStaticTypeResolver extends JavaTemplateEntity {

  private static final String GET_SCHEMA_PATH_METHOD_NAME = "getSchemaPath";
  private static final String GET_CATEGORY_NAME_METHOD_NAME = "getCategoryName";
  private static final String GET_QNAME_METHOD_NAME = "getQName";
  private static final String GET_FORMAT_METHOD_NAME = "getFormat";

  private final boolean requiresTypeResolver;
  private final boolean requiresToWriteSchema;

  private static final String SCHEMAS_FOLDER = "schemas";
  protected final TypeDefinition typeDefinition;
  private String schemaPath;

  private String className;
  private String javaPackage;
  private String categoryName;
  protected Class<?> superclass;

  /***
   * Creates a type resolver that will generate a schema file in the resources directory if needed.
   * @param sdkConnector the parent SdkConnector for this type
   * @param typeDefinition The type definition which this metadata will be created for
   * @param operation The operation this type definition will be generated for. It will be used to generate the schema name.
   * @param schemaNameType The type of name that this schema will get.
   * @param partName When this schema corresponds to a multipart part, this part name will be used to generate the schema name.
   *                 Must be an empty string if it he type definition does not correspond to a part.
   */
  public SdkAbstractStaticTypeResolver(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                       ConnectorOperation operation, TypeDefinition typeDefinition, SchemaNameType schemaNameType,
                                       String partName)
      throws TemplatingException {
    super(outputDir, connectorModel);

    this.typeDefinition = typeDefinition;
    superclass = buildSuperclass(typeDefinition);
    requiresTypeResolver = buildRequiresTypeResolver();
    requiresToWriteSchema = requiresToWriteSchema();


    if (requiresTypeResolver) {
      className = buildClassName(operation, partName);
      javaPackage = buildPackage(connectorModel);
      categoryName = buildCategoryName(operation);

      if (requiresToWriteSchema) {
        Path schemasFolder = sdkConnector.getResourcesPath().resolve(SCHEMAS_FOLDER);

        schemaPath = "/" + SCHEMAS_FOLDER + "/" +
            writeSchema(typeDefinition.getTypeSchema(), schemasFolder, operation, schemaNameType, partName,
                        sdkConnector.getTypeSchemaPaths());
      }
    }
  }

  protected abstract Class<?> buildSuperclass(TypeDefinition typeDefinition) throws TemplatingException;

  protected abstract String getClassNameSuffix();

  private String buildClassName(ConnectorOperation operation, String partName) {
    return getJavaUpperCamelNameFromXml(operation.getInternalName())
        + (isNotBlank(partName) ? getJavaUpperCamelNameFromXml(getXmlName(partName)) : EMPTY)
        + getClassNameSuffix();
  }

  private static String buildCategoryName(ConnectorOperation operation) {
    return operation.getInternalName() + "-type-resolver";
  }

  private String buildPackage(ConnectorModel connectorModel) {
    return connectorModel.getBasePackage() + ".internal.operation.metadata";
  }

  public String getClassName() {
    return className;
  }

  public String getPackage() {
    return javaPackage;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (requiresTypeResolver) {
      generateTypeResolverClass();
    }
  }

  private void generateTypeResolverClass() throws TemplatingException {
    TypeSpec.Builder typeResolverClassBuilder =
        TypeSpec
            .classBuilder(className)
            .addModifiers(Modifier.PUBLIC)
            .superclass(superclass);

    if (requiresToWriteSchema) {
      addGetSchemaPathMethod(typeResolverClassBuilder);
    }

    if (requiresCategoryNameMethod()) {
      addGetCategoryNameMethod(typeResolverClassBuilder);
    }

    if (requiresQNameMethod()) {
      addGetQNameMethod(typeResolverClassBuilder);
    }

    if (requiresFormatMethod()) {
      addFormatMethod(typeResolverClassBuilder);
    }

    writeClassToFile(typeResolverClassBuilder.build(), javaPackage);
  }

  private void addFormatMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return new $1T($2S, $2S, $2S)",
                      MetadataFormat.class,
                      typeDefinition.getMediaType().getType() + "/" + typeDefinition.getMediaType().getSubtype())
        .build();

    MethodSpec createConnectionMethod = MethodSpec.methodBuilder(GET_FORMAT_METHOD_NAME)
        .returns(TypeName.get(MetadataFormat.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(createConnectionMethod);
  }

  private void addGetSchemaPathMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return $S", schemaPath)
        .build();

    MethodSpec createConnectionMethod = MethodSpec.methodBuilder(GET_SCHEMA_PATH_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(createConnectionMethod);
  }

  private void addGetCategoryNameMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return $S", categoryName)
        .build();

    MethodSpec getCategoryNameMethod = MethodSpec.methodBuilder(GET_CATEGORY_NAME_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(getCategoryNameMethod);
  }

  private void addGetQNameMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return $S", getQName())
        .build();

    MethodSpec getCategoryNameMethod = MethodSpec.methodBuilder(GET_QNAME_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(getCategoryNameMethod);
  }

  protected boolean requiresCategoryNameMethod() {
    return true;
  }

  protected abstract boolean requiresQNameMethod();

  protected abstract boolean requiresFormatMethod();

  protected abstract String getQName();

  protected boolean buildRequiresTypeResolver() {
    return !(typeDefinition instanceof EmptyTypeDefinition);
  }

  public boolean getRequiresTypeResolver() {
    return requiresTypeResolver;
  }

  protected abstract boolean requiresToWriteSchema();
}
