package com.mule.connectors.interop;

import com.mule.connectors.interop.exceptions.CreatorMojoException;
import com.mule.connectors.interop.exceptions.ManifestUpdateException;
import com.mule.connectors.interop.exceptions.TargetPlatformUpdateException;
import com.mule.connectors.interop.model.Manifest;
import com.mule.connectors.interop.model.TargetPlatform;
import com.mule.connectors.interop.model.UpdateSite;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;


/**
 * @goal create
 * @phase package
 * @requiresProject false
 */
public class RunnerCreatorMojo extends AbstractMojo {

    private static final String BUNDLE_VERSION_0_0_0 = ";bundle-version=\"0.0.0\"\n";

    private String zipName = "interop-ce-project.zip";

    /**
     * @parameter expression="${externalPlatform}" default-value=""
     * */
    private String externalPlatform;

    /**
     * @parameter default-value="interop-ce-project/ce-interop-testsuite/META-INF/MANIFEST.MF"
     * */
    private String manifestPath;

    /**
     * @parameter default-value="interop-ce-project/ce-interop-target-platform/ce-interop-target-platform.target"
     * */
    private String platformPath;

    /**
     * Target directory present in root build directory
     * @parameter default-value="${project.build.directory}"
     * */
    private String targetDir;

    /**
     * Build root directory
     * @parameter default-value="${project.basedir}"
     * */
    private String baseDir;


    private Manifest manifest;
    private UpdateSite connectorUpdateSite;
    private TargetPlatform targetPlatform;

    public RunnerCreatorMojo(){}

    public RunnerCreatorMojo(String baseDir, String targetDir, String zipPath) {
        this.baseDir = baseDir;
        this.targetDir = targetDir;
        this.zipName = zipPath;
    }


    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        unzipFile(zipName);

        initilizeRunnerFiles();

        updateManifest();

        updateTargetPlatform();
    }

    private void updateManifest() {

        final String bundleLine = " " + connectorUpdateSite.getManifestUnit().getId() + BUNDLE_VERSION_0_0_0;

        try {
            manifest.updateRequiredBundle(bundleLine);
        } catch (ManifestUpdateException e) {
            e.printStackTrace();
            throw new CreatorMojoException(e.getMessage());
        }
    }

    private void updateTargetPlatform() {

        try {
            if (!StringUtils.isEmpty(externalPlatform) && new File(externalPlatform).exists()){
                replaceTargetPlatform();
            }
            targetPlatform.addLocation(connectorUpdateSite);
        } catch (IOException e) {
            e.printStackTrace();
            throw new CreatorMojoException(e.getMessage());
        }
    }

    private void replaceTargetPlatform() throws IOException {
        FileUtils.forceDelete(new File(targetDir, platformPath));
        FileUtils.copyFile(new File(externalPlatform), new File(targetDir, platformPath));
        initializeTargetPlatform();
    }

    public void unzipFile(String zipFile){
        try{
            //create output directory is not exists
            File folder = new File(targetDir);
            if(!folder.exists()){
                folder.mkdir();
            }
            getLog().debug("Unzip directory :: " + folder.getCanonicalPath());
            //get the zip file content
            ZipInputStream zis = new ZipInputStream(this.getClass().getClassLoader().getResourceAsStream(zipFile));
            //get the zipped file list entry
            ZipEntry ze = zis.getNextEntry();

            byte[] buffer = new byte[1024];
            while(ze!=null){
                String fileName = ze.getName();
                File newFile = new File(targetDir, fileName);
                new File(newFile.getParent()).mkdirs();

                if (!ze.isDirectory()){
                    getLog().debug("File unzip : "+ newFile.getAbsoluteFile());
                    FileOutputStream fos = new FileOutputStream(newFile);

                    int len;
                    while ((len = zis.read(buffer)) > 0) {
                        fos.write(buffer, 0, len);
                    }
                    fos.flush();
                    fos.close();
                }
                ze = zis.getNextEntry();
            }
            zis.closeEntry();
            zis.close();
            getLog().debug("Done");

        }catch(IOException e){
            e.printStackTrace();
            throw new CreatorMojoException(e.getMessage());
        }
    }

    public void setTargetDir(String targetDir) {
        this.targetDir = targetDir;
    }

    public void setBaseDir(String baseDir) {
        this.baseDir = baseDir;
    }

    public void setManifestPath(String manifestPath) {
        this.manifestPath = manifestPath;
    }

    public void setOutputDir(String targetDir) {
        this.targetDir = targetDir;
    }

    private void initilizeRunnerFiles() {
        try {
            initializeManifest();

            initializeTargetPlatform();

            initializeUpdateSite();

        } catch (IOException e) {
            e.printStackTrace();
            throw new CreatorMojoException(e.getMessage());
        }
    }

    private void initializeUpdateSite() throws IOException {
        connectorUpdateSite = new UpdateSite(targetDir);
    }

    private void initializeTargetPlatform() throws TargetPlatformUpdateException {
        targetPlatform = new TargetPlatform(new File(targetDir, platformPath).getPath());
    }

    private void initializeManifest() {
        manifest = new Manifest(new File(targetDir, manifestPath).getPath());
    }

    public void setPlatformPath(String platformPath) {
        this.platformPath = platformPath;
    }

    public void setExternalPlatform(String externalPlatform) {
        this.externalPlatform = externalPlatform;
    }
}
