package com.mule.connectors.interop.model;

import com.mule.connectors.interop.exceptions.TargetPlatformUpdateException;
import org.apache.commons.io.FileUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.File;
import java.io.IOException;

/**
 * @author Mulesoft, Inc
 */
public class TargetPlatform {

    private final XPath xpath;
    private File platform;
    private Document doc;

    public TargetPlatform(String platformPath) throws TargetPlatformUpdateException {
        try {
            platform = new File(platformPath);

            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            doc = builder.parse(platform);

            XPathFactory xPathfactory = XPathFactory.newInstance();
            xpath = xPathfactory.newXPath();

        } catch (ParserConfigurationException e) {
            e.printStackTrace();
            throw new TargetPlatformUpdateException(e.getMessage());
        } catch (SAXException e) {
            e.printStackTrace();
            throw new TargetPlatformUpdateException(e.getMessage());
        } catch (IOException e) {
            e.printStackTrace();
            throw new TargetPlatformUpdateException(e.getMessage());
        }
    }

    public void updateLocation(String oldLocation, String newLocation) throws XPathExpressionException, TransformerException, IOException {

        final String studioPattern = "//repository[contains(@location, '" + oldLocation + "')]";
        Element studioRepository = (Element) xpath.compile(studioPattern).evaluate(doc.getDocumentElement(), XPathConstants.NODE);
        studioRepository.setAttribute("location", newLocation);

        exportToFile(doc);
    }

    public void addLocation(UpdateSite updateSite) throws TargetPlatformUpdateException {

        try {
            if (!containsUnit(updateSite.getPlatformUnit().getId())) {

                Element location = createNewLocation(doc.getDocumentElement());

                Element unit = doc.createElement("unit");
                unit.setAttribute("id", updateSite.getPlatformUnit().getId());
                unit.setAttribute("version", updateSite.getPlatformUnit().getVersion());

                location.appendChild(unit);

                Element repository = doc.createElement("repository");
                repository.setAttribute("location", updateSite.getLocation());

                location.appendChild(repository);

                exportToFile(doc);
            }

        } catch (XPathExpressionException e) {
            e.printStackTrace();
            throw new TargetPlatformUpdateException(e.getMessage());
        } catch (TransformerException e) {
            e.printStackTrace();
            throw new TargetPlatformUpdateException(e.getMessage());
        } catch (IOException e) {
            e.printStackTrace();
            throw new TargetPlatformUpdateException(e.getMessage());
        }
    }

    private boolean containsUnit(String unit) throws IOException {

        return FileUtils.readLines(platform).contains(unit);
    }

    private Element createNewLocation(Element target) throws XPathExpressionException {

        Element newLocation = doc.createElement("location");
        newLocation.setAttribute("includeAllPlatforms", "false");
        newLocation.setAttribute("includeConfigurePhase", "true");
        newLocation.setAttribute("includeMode", "planner");
        newLocation.setAttribute("includeSource", "true");
        newLocation.setAttribute("type", "InstallableUnit");

        Element locations = (Element) xpath.compile("./locations").evaluate(target, XPathConstants.NODE);
        locations.appendChild(newLocation);

        return newLocation;
    }

    private void exportToFile(Document doc) throws TransformerException, IOException {

        TransformerFactory transformerFactory = TransformerFactory.newInstance();
        Transformer transformer = transformerFactory.newTransformer();
        DOMSource source = new DOMSource(doc);
        File newPlatform = new File(platform.getCanonicalPath() + ".tmp");

        StreamResult result = new StreamResult(newPlatform);

        transformer.setOutputProperty(OutputKeys.INDENT, "yes");
        transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "0");

        transformer.transform(source, result);

        FileUtils.forceDelete(platform);
        FileUtils.moveFile(newPlatform, platform);
    }

}

