package com.mule.connectors.interop;

import com.mule.connectors.interop.exception.TestDataParserException;
import com.mule.connectors.interop.exceptions.CreatorMojoException;
import com.mule.connectors.interop.exceptions.ManifestUpdateException;
import com.mule.connectors.interop.exceptions.TargetPlatformUpdateException;
import com.mule.connectors.interop.model.*;
import com.mule.connectors.interop.parser.TestDataParser;
import com.mule.connectors.testdata.model.TestData;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import static com.mule.connectors.interop.Utils.unzipFile;


@Mojo( name = "create")
@Execute( goal = "create")
public class RunnerCreatorMojo extends AbstractMojo {

    private static final String BUNDLE_VERSION_0_0_0 = ";bundle-version=\"0.0.0\"\n";

    private String zipName = "interop-ce-project.zip";

    @Parameter( property="externalPlatform", defaultValue="")
    private String externalPlatform;

    @Parameter( defaultValue="interop-ce-project/ce-interop-testsuite/META-INF/MANIFEST.MF")
    private String manifestPath;


    @Parameter( defaultValue="interop-ce-project/ce-interop-target-platform/ce-interop-target-platform.target")
    private String platformPath;

    /**
     * Target directory present in root build directory
     * */
     @Parameter( defaultValue = "${project.build.directory}", readonly = true )
     private String targetDir;

    @Parameter( defaultValue = "${project.basedir}", readonly = true )
    protected File baseDir;


    private ManifestFile manifest;
    private UpdateSite connectorUpdateSite;
    private TargetPlatform targetPlatform;
    private DependenciesPom dependenciesPom;

    public RunnerCreatorMojo(){}

    public RunnerCreatorMojo(File baseDir, String targetDir, String zipPath) {
        this.baseDir = baseDir;
        this.targetDir = targetDir;
        this.zipName = StringUtils.isEmpty(zipPath) ? zipName : zipPath;
    }


    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        unzipFile(targetDir, zipName);

        initilizeRunnerFiles();

        updateManifest();

        updateTargetPlatform();

        TestDataFile overrides = new TestDataFile(baseDir, RunExecutorMojo.testDataOverrideNameFormat);
        updateDependenciesPom(overrides.getPath());
    }

    private void updateDependenciesPom(String overridesFilePath) throws CreatorMojoException {

        try {
            TestData testDataModel = TestDataParser.unmarshall(new FileInputStream(new File(overridesFilePath)));
            if (testDataModel.getProperties() != null && testDataModel.getProperties().getDependencies() !=  null) {
                this.dependenciesPom.updateDependencies(testDataModel.getProperties().getDependencies());
            }

        } catch (IOException e) {
            e.printStackTrace();
        } catch (TestDataParserException e) {
            e.printStackTrace();
        }
    }

    private void updateManifest() {

        final String bundleLine = " " + connectorUpdateSite.getManifestUnit().getId() + BUNDLE_VERSION_0_0_0;

        try {
            manifest.updateRequiredBundle(bundleLine);
        } catch (ManifestUpdateException e) {
            e.printStackTrace();
            throw new CreatorMojoException(e.getMessage());
        }
    }

    private void updateTargetPlatform() {

        try {
            if (!StringUtils.isEmpty(externalPlatform) && new File(externalPlatform).exists()){
                replaceTargetPlatform();
            }
            targetPlatform.addLocation(connectorUpdateSite);
        } catch (IOException e) {
            e.printStackTrace();
            throw new CreatorMojoException(e.getMessage());
        }
    }

    private void replaceTargetPlatform() throws IOException {
        FileUtils.forceDelete(new File(targetDir, platformPath));
        FileUtils.copyFile(new File(externalPlatform), new File(targetDir, platformPath));
        initializeTargetPlatform();
    }


    public void setTargetDir(String targetDir) {
        this.targetDir = targetDir;
    }

    public void setBaseDir(File baseDir) {
        this.baseDir = baseDir;
    }

    public void setManifestPath(String manifestPath) {
        this.manifestPath = manifestPath;
    }

    public void setOutputDir(String targetDir) {
        this.targetDir = targetDir;
    }

    private void initilizeRunnerFiles() {
        try {
            initializeManifest();

            initializeTargetPlatform();

            initializeUpdateSite();

            initializeDependenciesPom();

        } catch (IOException e) {
            e.printStackTrace();
            throw new CreatorMojoException(e.getMessage());
        }
    }

    private void initializeUpdateSite() throws IOException {
        connectorUpdateSite = new UpdateSite(targetDir);
    }

    private void initializeTargetPlatform() throws TargetPlatformUpdateException {
        targetPlatform = new TargetPlatform(new File(targetDir, platformPath).getPath());
    }

    private void initializeManifest() {
        manifest = new ManifestFile(new File(targetDir, manifestPath).getPath());
    }

    public void setPlatformPath(String platformPath) {
        this.platformPath = platformPath;
    }

    public void setExternalPlatform(String externalPlatform) {
        this.externalPlatform = externalPlatform;
    }

    public void initializeDependenciesPom() {
        dependenciesPom = new DependenciesPom(new File(targetDir, DependenciesPom.DEPENDENCIES_POM_PATH));
    }
}
