/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.connection.datasource;

import org.mule.db.commons.api.config.DbPoolingProfile;
import org.mule.db.commons.api.param.ColumnType;
import org.mule.db.commons.internal.domain.connection.DataSourceConfig;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.connection.DbConnectionProvider;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;

import javax.sql.DataSource;
import java.util.List;
import java.util.Optional;

/**
 * {@link ConnectionProvider} implementation which creates DB connections from a referenced {@link
 * DataSource}
 * Consumers of {@link DataSourceReferenceConnectionProvider} must create a class similar to this one with the appropriate annotations
 * (DisplayName, Alias, ExternalLib, ParameterGroup, others), and have an instance of this one to pass the required
 * properties.
 */
public class DataSourceReferenceConnectionProvider implements ConnectionProvider<DbConnection>, Initialisable, Disposable {

  private DbConnectionProvider dbConnectionProvider;
  private final String configName;
  private final Registry registry;
  private final DbPoolingProfile poolingProfile;
  private final List<? extends ColumnType> columnTypes;
  private final DataSourceConnectionSettings connectionSettings;

  public DataSourceReferenceConnectionProvider(String configName, Registry registry, DbPoolingProfile poolingProfile,
                                               List<? extends ColumnType> columnTypes,
                                               DataSourceConnectionSettings connectionSettings) {
    this.configName = configName;
    this.registry = registry;
    this.poolingProfile = poolingProfile;
    this.columnTypes = columnTypes;
    this.connectionSettings = connectionSettings;
  }

  @Override
  public void initialise() throws InitialisationException {
    dbConnectionProvider = new DbConnectionProvider(configName, registry, poolingProfile, columnTypes) {

      @Override
      public Optional<DataSource> getDataSource() {
        return Optional.ofNullable(connectionSettings.getDataSourceRef());
      }

      @Override
      public Optional<DataSourceConfig> getDataSourceConfig() {
        return Optional.empty();
      }
    };

    dbConnectionProvider.initialise();
  }

  @Override
  public void dispose() {
    dbConnectionProvider.dispose();
  }

  @Override
  public DbConnection connect() throws ConnectionException {
    return dbConnectionProvider.connect();
  }

  @Override
  public void disconnect(DbConnection dbConnection) {
    dbConnectionProvider.disconnect(dbConnection);
  }

  @Override
  public ConnectionValidationResult validate(DbConnection dbConnection) {
    return dbConnectionProvider.validate(dbConnection);
  }

}
