/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.connection.generic;

import org.mule.db.commons.api.config.DbPoolingProfile;
import org.mule.db.commons.api.param.ColumnType;
import org.mule.db.commons.internal.domain.connection.DataSourceConfig;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.connection.DbConnectionProvider;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;

import java.util.List;
import java.util.Optional;
import javax.sql.DataSource;

/**
 * {@link ConnectionProvider} that creates connections for any kind of database using a JDBC URL and the required JDBC Driver
 * Class. Consumers of {@link GenericConnectionProvider} must create a class similar to this one with the appropriate annotations
 * (DisplayName, Alias, ExternalLib, ParameterGroup, others), and have an instance of this one to pass the required properties.
 */
public class GenericConnectionProvider implements ConnectionProvider<DbConnection>, Initialisable, Disposable {

  private DbConnectionProvider dbConnectionProvider;
  private final String configName;
  private final Registry registry;
  private final DbPoolingProfile poolingProfile;
  private final List<? extends ColumnType> columnTypes;
  private final DataSourceConfig connectionParameters;

  public GenericConnectionProvider(String configName, Registry registry, DbPoolingProfile poolingProfile,
                                   List<? extends ColumnType> columnTypes, DataSourceConfig connectionParameters) {
    this.configName = configName;
    this.registry = registry;
    this.poolingProfile = poolingProfile;
    this.columnTypes = columnTypes;
    this.connectionParameters = connectionParameters;
  }

  @Override
  public void initialise() throws InitialisationException {
    dbConnectionProvider = new DbConnectionProvider(configName, registry, poolingProfile, columnTypes) {

      @Override
      public Optional<DataSource> getDataSource() {
        return Optional.empty();
      }

      @Override
      public Optional<DataSourceConfig> getDataSourceConfig() {
        return Optional.ofNullable(connectionParameters);
      }
    };

    dbConnectionProvider.initialise();
  }

  @Override
  public void dispose() {
    dbConnectionProvider.dispose();
  }

  @Override
  public DbConnection connect() throws ConnectionException {
    return dbConnectionProvider.connect();
  }

  @Override
  public void disconnect(DbConnection dbConnection) {
    dbConnectionProvider.disconnect(dbConnection);
  }

  @Override
  public ConnectionValidationResult validate(DbConnection dbConnection) {
    return dbConnectionProvider.validate(dbConnection);
  }

}
