/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.resolver.query;

import org.mule.db.commons.api.param.ParameterizedStatementDefinition;
import org.mule.db.commons.internal.domain.query.QueryParamValue;
import org.mule.db.commons.internal.domain.query.QueryTemplate;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.util.List;
import java.util.Optional;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

public class ParameterizedQueryResolver<T extends ParameterizedStatementDefinition<?>> extends AbstractQueryResolver<T> {

  @Override
  protected List<QueryParamValue> resolveParams(T statementDefinition, QueryTemplate template, StreamingHelper streamingHelper) {
    return template.getInputParams().stream()
        .map(p -> {
          final String parameterName = p.getName();

          Optional<Reference<Object>> parameterValue =
              getStreamingAwareParameter(statementDefinition, parameterName, streamingHelper);
          if (parameterValue.isPresent()) {
            return new QueryParamValue(parameterName, parameterValue.get().get());
          } else {
            throw new IllegalArgumentException(format("Parameter '%s' was not bound for query '%s'",
                                                      parameterName, statementDefinition.getSql()));
          }
        }).collect(toList());
  }

  private Optional<Reference<Object>> getStreamingAwareParameter(T statementDefinition, String parameterName,
                                                                 StreamingHelper streamingHelper) {
    return getInputParameter(statementDefinition, parameterName)
        .map(ref -> streamingHelper != null ? new Reference<>(streamingHelper.resolveCursor(ref.get())) : ref);
  }

  protected Optional<Reference<Object>> getInputParameter(T statementDefinition, String parameterName) {
    return statementDefinition.getInputParameter(parameterName);
  }
}
