/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.connection;

import org.mule.db.commons.api.exception.connection.ConnectionCreationException;
import org.mule.db.commons.internal.domain.type.DbType;
import org.mule.db.commons.internal.domain.type.MappedStructResolvedDbType;
import org.mule.db.commons.internal.util.CredentialsMaskUtils;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import javax.sql.DataSource;

import static java.util.Optional.of;

/**
 * A factor for JDBC {@link Connection}s
 *
 * @since 1.0
 */
public class JdbcConnectionFactory {

  private final Function<String, String> maskCredentialsFunction;

  protected JdbcConnectionFactory(Builder builder) {
    this.maskCredentialsFunction = builder.maskCredentialsFunctionOptional.get();
  }

  /**
   * Creates a new JDBC {@link Connection}
   *
   * @param dataSource      the {@link DataSource} from which the connection comes from
   * @param customDataTypes user defined data types
   * @return a {@link Connection}
   */
  public Connection createConnection(DataSource dataSource, List<DbType> customDataTypes)
      throws SQLException, ConnectionCreationException {
    try {
      Connection connection;
      connection = dataSource.getConnection();

      if (connection == null) {
        throw new ConnectionCreationException("Unable to create connection to the provided dataSource: " + dataSource);
      }

      Map<String, Class<?>> typeMapping = createTypeMapping(customDataTypes);

      if (typeMapping != null && !typeMapping.isEmpty()) {
        connection.setTypeMap(typeMapping);
      }

      return connection;
    } catch (SQLException e) {
      String s = maskCredentialsFunction.apply(e.getMessage());
      throw new SQLException(s, e.getSQLState(), e.getErrorCode(), e.getCause());
    }
  }

  private Map<String, Class<?>> createTypeMapping(List<DbType> customDataTypes) {
    final Map<String, Class<?>> typeMapping = new HashMap<>();

    customDataTypes.stream()
        .filter(dbType -> dbType instanceof MappedStructResolvedDbType)
        .forEach(dbType -> {
          final MappedStructResolvedDbType structDbType = (MappedStructResolvedDbType) dbType;
          if (structDbType.getMappedClass() != null) {
            typeMapping.put(structDbType.getName(), structDbType.getMappedClass());
          }
        });

    return typeMapping;
  }

  public static class Builder {

    private Optional<Function<String, String>> maskCredentialsFunctionOptional = Optional.empty();

    public JdbcConnectionFactory.Builder withMaskCredentialsFunction(Function<String, String> maskCredentialsFunction) {
      this.maskCredentialsFunctionOptional = of(maskCredentialsFunction);
      return this;
    }

    public JdbcConnectionFactory build() {
      if (!maskCredentialsFunctionOptional.isPresent())
        this.maskCredentialsFunctionOptional = of(CredentialsMaskUtils::maskUrlUserAndPassword);
      return new JdbcConnectionFactory(this);
    }

  }

}
