/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.exception;

import org.mule.db.commons.api.exception.connection.BadSqlSyntaxException;
import org.mule.db.commons.api.exception.connection.QueryExecutionException;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.exception.ExceptionHandler;

import java.sql.SQLException;
import java.sql.SQLSyntaxErrorException;

/**
 * Translates {@link SQLException} into connector specific ones.
 *
 * @since 1.0
 */
// TODO - MULE-13798: Improve DB Error types by using SQLException hierarchy and SQLState
public class DbExceptionHandler extends ExceptionHandler {

  @Override
  public Exception enrichException(Exception e) {
    if (e instanceof ModuleException) {
      return e;
    }

    return getCauseOfType(e, SQLSyntaxErrorException.class)
        .map(cause -> (Exception) new BadSqlSyntaxException(e.getMessage(), e))
        .orElseGet(() -> getCauseOfType(e, SQLException.class)
            .map(sqlException -> {
              if (isConnectionException(sqlException) || e instanceof ConnectionException) {
                return new ConnectionException(sqlException.getMessage(), sqlException);
              }

              if (isBadSyntaxException(sqlException)) {
                return new BadSqlSyntaxException(sqlException.getMessage(), sqlException);
              }

              return new QueryExecutionException(sqlException.getMessage(), sqlException);
            })
            .orElse(e));
  }

  private boolean isConnectionException(SQLException e) {
    String sqlState = e.getSQLState();
    return "08S01".equals(sqlState) || "08001".equals(sqlState);
  }

  private boolean isBadSyntaxException(SQLException e) {
    String sqlState = e.getSQLState();
    return "S0001".equals(sqlState);
  }

}
