/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.db.commons.internal.domain.connection;

import static java.util.concurrent.ConcurrentHashMap.newKeySet;

import org.mule.db.commons.api.config.DbPoolingProfile;
import org.mule.db.commons.internal.domain.xa.CompositeDataSourceDecorator;
import org.mule.runtime.api.config.DatabasePoolingProfile;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.tx.DataSourceDecorator;

import com.mchange.v2.c3p0.DataSources;

import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.sql.DataSource;

import org.enhydra.jdbc.standard.StandardDataSource;
import org.enhydra.jdbc.standard.StandardXADataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Creates {@link DataSource} instances
 */
public class DataSourceFactory implements Disposable {

  private static final Logger LOGGER = LoggerFactory.getLogger(DataSourceFactory.class);

  private final String name;
  private final Set<DataSource> pooledDataSources = newKeySet();
  private final Set<Disposable> disposableDataSources = newKeySet();
  private final CompositeDataSourceDecorator dataSourceDecorator;

  public DataSourceFactory(String name, Collection<DataSourceDecorator> dataSourceDecorators) {
    this.name = name;
    dataSourceDecorator = new CompositeDataSourceDecorator(dataSourceDecorators);
  }

  /**
   * Creates a dataSource from a given dataSource config
   *
   * @param dataSourceConfig describes how to create the dataSource
   * @return a non null dataSource
   * @throws SQLException in case there is a problem creating the dataSource
   */
  public DataSource create(DataSourceConfig dataSourceConfig, DbPoolingProfile poolingProfile) throws SQLException {
    DataSource dataSource;

    if (poolingProfile == null) {
      dataSource = createSingleDataSource(dataSourceConfig);
    } else {
      dataSource = createPooledDataSource(dataSourceConfig, poolingProfile);
    }

    if (dataSourceConfig.isUseXaTransactions()) {
      dataSource = decorateDataSource(dataSource, poolingProfile);
    }

    if (!(poolingProfile == null || dataSourceConfig.isUseXaTransactions())) {
      pooledDataSources.add(dataSource);
    } else if (dataSource instanceof Disposable) {
      disposableDataSources.add((Disposable) dataSource);
    }

    return dataSource;
  }

  public DataSource decorateDataSource(DataSource dataSource, DatabasePoolingProfile poolingProfile) {
    return dataSourceDecorator.decorate(dataSource, name, poolingProfile);
  }

  protected DataSource createSingleDataSource(DataSourceConfig dataSourceConfig) throws SQLException {
    StandardDataSource dataSource =
        dataSourceConfig.isUseXaTransactions() ? new StandardXADataSource() : new StandardDataSource();
    dataSource.setDriverName(dataSourceConfig.getDriverClassName());
    dataSource.setPassword(dataSourceConfig.getPassword());
    dataSource.setTransactionIsolation(dataSourceConfig.getTransactionIsolation().getCode());
    dataSource.setUrl(dataSourceConfig.getUrl());
    dataSource.setUser(dataSourceConfig.getUser());

    return dataSource;
  }

  protected DataSource createPooledDataSource(DataSourceConfig dataSourceConfig, DbPoolingProfile poolingProfile)
      throws SQLException {
    if (dataSourceConfig.isUseXaTransactions()) {
      return createSingleDataSource(dataSourceConfig);
    } else {
      return createPooledStandardDataSource(createSingleDataSource(dataSourceConfig), poolingProfile);
    }
  }

  protected DataSource createPooledStandardDataSource(DataSource dataSource, DbPoolingProfile poolingProfile)
      throws SQLException {
    Map<String, Object> config = new HashMap<>();
    config.put("maxPoolSize", poolingProfile.getMaxPoolSize());
    config.put("minPoolSize", poolingProfile.getMinPoolSize());
    config.put("initialPoolSize", poolingProfile.getMinPoolSize());
    config.put("checkoutTimeout", new Long(poolingProfile.getMaxWaitUnit().toMillis(poolingProfile.getMaxWait())).intValue());
    config.put("acquireIncrement", poolingProfile.getAcquireIncrement());
    config.put("maxStatements", poolingProfile.getMaxStatements());
    config.put("maxIdleTime", poolingProfile.getMaxIdleTime());
    config.put("testConnectionOnCheckout", poolingProfile.getTestConnectionOnCheckout());
    config.put("maxStatementsPerConnection", poolingProfile.getPreparedStatementCacheSize());
    poolingProfile.getAdditionalProperties().entrySet().forEach((param) -> {
      if (config.containsKey(param.getKey()) && !config.get(param.getKey()).equals(param.getValue())) {
        LOGGER.warn("Attempted to override property {0} using additional-properties. Proceeding to use {0} = {1}.",
                    param.getKey(), param.getValue());
      } else {
        config.put(param.getKey(), param.getValue());
      }
    });
    return DataSources.pooledDataSource(dataSource, config);
  }

  @Override
  public void dispose() {
    for (DataSource pooledDataSource : pooledDataSources) {
      try {
        DataSources.destroy(pooledDataSource);
      } catch (SQLException e) {
        LOGGER.warn("Unable to properly release pooled data source", e);
      }
    }

    for (Disposable disposableDataSource : disposableDataSources) {
      try {
        disposableDataSource.dispose();
      } catch (Exception e) {
        LOGGER.warn("Unable to properly dispose data source", e);
      }
    }
  }
}
