/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal;

import org.mule.db.commons.api.DbAggregate;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.core.api.extension.ExtensionManager;

import java.sql.Array;
import java.sql.Struct;
import java.util.List;
import javax.inject.Inject;

/**
 * Database connector functions to create complex JDBC Structures.
 *
 * @since 1.5.1
 */
public class DbFunctions implements Initialisable {

  @Inject
  ExtensionManager extensionManager;

  DbFunctionUtil util = null;

  /**
   * DataWeave function to create JDBC Array objects based on the Array Type to create and the values that conforms the type.
   *
   * @param typeName   The name of the Array type to create
   * @param values     An array of values that conforms the Array Type
   * @param configName The configuration in charge of creating the Array Type
   * @return A JDBC Array
   */
  @SuppressWarnings("unused")
  public Object createArray(String configName, String typeName, List<Object> values) {
    try {
      return util.execute((con) -> con.createArray(typeName, values.toArray()), configName);
    } catch (Throwable t) {
      throw new RuntimeException("An error occurred when trying to create SQL array \"" + typeName + "\":  " + t, t);
    }
  }

  /**
   * DataWeave function to prepare an array for sending to the database as an {@link Array}.
   *
   * This function returns an object that will later be replaced with an {@link Array} when passed as a parameter to a database
   * operation.
   *
   * @param typeName The name of the Array type to create
   * @param values   An array of values that conforms the Array Type
   * @return An object holding all the information for creating an Array later
   */
  @SuppressWarnings("unused")
  public Object prepareArray(String typeName, List<Object> values) {
    return new DbAggregate(DbAggregate.Kind.ARRAY, typeName, values);
  }

  /**
   * DataWeave function to create JDBC Struct objects based on the Type Name and their correspondent properties.
   *
   * @param typeName   The name of the Struct type to create
   * @param properties An array of values that conforms the Struct properties
   * @param configName The configuration in charge of creating the Struct type
   * @return A JDBC Struct
   */
  @SuppressWarnings("unused")
  public Object createStruct(String configName, String typeName, List<Object> properties) {
    try {
      return util.execute((con) -> con.createStruct(typeName, properties.toArray()), configName);
    } catch (Throwable t) {
      throw new RuntimeException("An error occurred when trying to create SQL struct \"" + typeName + "\":  "
          + t, t);
    }
  }

  /**
   * DataWeave function to prepare an array for sending to the database as a {@link Struct}.
   *
   * This function returns an object that will later be replaced with a {@link Struct} when passed as a parameter to a database
   * operation.
   *
   * @param typeName   The name of the Struct type to create
   * @param properties An array of values that conforms the Struct properties
   * @return An object holding all the information for creating a Struct later
   */
  @SuppressWarnings("unused")
  public Object prepareStruct(String typeName, List<Object> properties) {
    return new DbAggregate(DbAggregate.Kind.STRUCT, typeName, properties);
  }

  @Override
  public void initialise() throws InitialisationException {
    util = new DbFunctionUtil(extensionManager);
  }

}
