/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.metadata;

import static java.sql.Types.VARCHAR;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_CONFIGURATION;
import static org.mule.runtime.api.metadata.resolving.FailureCode.UNKNOWN;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.param.InputQueryParam;
import org.mule.db.commons.internal.domain.query.QueryTemplate;
import org.mule.db.commons.internal.parser.SimpleQueryTemplateParser;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.metadata.MetadataContext;
import org.mule.runtime.api.metadata.MetadataResolvingException;
import org.mule.runtime.api.metadata.resolving.InputTypeResolver;
import org.slf4j.Logger;

import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DbInputMetadataResolver extends BaseDbMetadataResolver implements InputTypeResolver<String> {

  public static final Logger LOGGER = getLogger(DbInputMetadataResolver.class);

  @Override
  public String getCategoryName() {
    return "DbCategory";
  }

  @Override
  public MetadataType getInputMetadata(MetadataContext context, String query)
      throws MetadataResolvingException, ConnectionException {

    this.typeLoader = context.getTypeLoader();
    this.typeBuilder = context.getTypeBuilder();

    QueryTemplate queryTemplate = parseQuery(query);
    List<InputQueryParam> inputParams = queryTemplate.getInputParams();
    // No metadata when no input parameters
    if (inputParams.isEmpty()) {
      return typeBuilder.nullType().build();
    }

    PreparedStatement statement = getStatement(context, queryTemplate);
    List<String> fieldNames = new ArrayList<>();
    for (InputQueryParam inputParam : inputParams) {
      String name = inputParam.getName();
      if (name == null) {
        return typeBuilder.anyType().build();
      }
      fieldNames.add(name);
    }

    try {
      return getInputMetadataUsingStatementMetadata(statement, fieldNames);
    } catch (SQLException e) {
      return getStaticInputMetadata(fieldNames);
    }
  }

  @Override
  protected QueryTemplate parseQuery(String query) {
    return new SimpleQueryTemplateParser().parse(query);
  }

  @Override
  protected PreparedStatement getStatement(MetadataContext context, QueryTemplate query)
      throws ConnectionException, MetadataResolvingException {
    DbConnection connection = context.<DbConnection>getConnection()
        .orElseThrow(() -> new MetadataResolvingException("A connection is required to resolve Metadata but none was provided",
                                                          INVALID_CONFIGURATION));
    PreparedStatement statement;
    try {
      statement = connection.getJdbcConnection().prepareStatement(query.getSqlText());
    } catch (SQLException e) {
      throw new MetadataResolvingException(e.getMessage(), UNKNOWN, e);
    }
    return statement;
  }

  private MetadataType getStaticInputMetadata(List<String> fieldNames) {
    Map<String, MetadataType> recordModels = new HashMap<>();

    for (String fieldName : fieldNames) {
      recordModels.put(fieldName, getDataTypeMetadataModel(VARCHAR));
    }

    ObjectTypeBuilder record = typeBuilder.objectType();
    recordModels.entrySet().forEach(e -> record.addField().key(e.getKey()).value(e.getValue()));
    return record.build();
  }

  private MetadataType getInputMetadataUsingStatementMetadata(PreparedStatement statement, List<String> fieldNames)
      throws SQLException {
    ParameterMetaData parameterMetaData = statement.getParameterMetaData();

    Map<String, MetadataType> recordModels = new HashMap<>();
    int i = 1;
    ObjectTypeBuilder record = typeBuilder.objectType();
    for (String fieldName : fieldNames) {
      int dataType = parameterMetaData.getParameterType(i);

      ObjectFieldTypeBuilder fieldTypeBuilder = record.addField();
      fieldTypeBuilder.key(fieldName);

      String parameterClassName = null;
      try {
        parameterClassName = parameterMetaData.getParameterClassName(i);
      } catch (Exception e) {
        LOGGER.debug("Could not get the class name for field name {} and data type id {}", fieldName, dataType);
      }
      try {
        if (parameterClassName != null) {
          fieldTypeBuilder.value(getDataTypeMetadataModel(dataType, parameterClassName));
        } else {
          fieldTypeBuilder.value(getDataTypeMetadataModel(dataType));
        }
      } catch (Exception e) {
        LOGGER.error("Could not map the data type for field name {}, data type id {} and parameter class name {}", fieldName,
                     dataType, parameterClassName);
        // If we fail to retrieve the MetadataType of the field we do a best effort and use AnyType
        fieldTypeBuilder.value(typeBuilder.anyType().build());
      }

      try {
        int nullableCode = parameterMetaData.isNullable(i);
        // Is not nulleable
        if (nullableCode == 0) {
          fieldTypeBuilder.required();
        }
      } catch (Exception e) {
        // If we fail to retrieve if the field is nullable, we do a best effort and asume it is nullable
      }

      i++;
    }

    recordModels.entrySet().forEach(e -> record.addField().key(e.getKey()).value(e.getValue()));
    return record.build();
  }
}
