/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.resolver.param;

import org.mule.db.commons.api.param.ParameterType;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.param.QueryParam;
import org.mule.db.commons.internal.domain.query.QueryTemplate;
import org.mule.db.commons.internal.domain.type.DbType;
import org.mule.db.commons.internal.domain.type.DbTypeManager;
import org.mule.db.commons.internal.domain.type.ResolvedDbType;
import org.mule.db.commons.internal.domain.type.UnknownDbType;
import org.mule.db.commons.internal.domain.type.UnknownDbTypeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static java.lang.Boolean.parseBoolean;
import static java.lang.String.format;
import static java.lang.String.join;
import static java.lang.System.getProperty;
import static org.apache.commons.lang3.StringUtils.substringAfterLast;
import static org.mule.db.commons.internal.util.StoredProcedureUtils.analyzeStoredOperation;
import static org.mule.db.commons.internal.util.StoredProcedureUtils.getStoreProcedureOwner;
import static org.mule.db.commons.internal.util.StoredProcedureUtils.getStoredProcedureParentOwner;

import org.mule.runtime.api.util.Pair;

/**
 * Resolves parameter types for stored procedure queries
 */
public class StoredProcedureParamTypeResolver implements ParamTypeResolver {

  private static final Logger LOGGER = LoggerFactory.getLogger(StoredProcedureParamTypeResolver.class);

  public static final String FORCE_SP_PARAM_TYPES = "mule.db.connector.force.sp.param.types";
  private static final int PARAM_NAME_COLUMN_INDEX = 4;
  private static final int TYPE_ID_COLUMN_INDEX = 6;
  private static final int TYPE_NAME_COLUMN_INDEX = 7;
  private static final int COLUMN_TYPE_INDEX = 5;
  private static final short PROCEDURE_COLUMN_RETURN_COLUMN_TYPE = 5;

  private final DbTypeManager dbTypeManager;

  public StoredProcedureParamTypeResolver(DbTypeManager dbTypeManager) {
    this.dbTypeManager = dbTypeManager;
  }

  @Override
  public Map<Integer, DbType> getParameterTypes(DbConnection connection, QueryTemplate queryTemplate,
                                                List<ParameterType> parameterTypesConfigured)
      throws SQLException {

    Map<Integer, DbType> parameters;
    if (shouldForceParametersTypes()) {
      parameters = getParameterTypesFromConfiguration(queryTemplate, parameterTypesConfigured);
      List<String> missingParameters = getMissingParameters(queryTemplate, parameters);
      if (missingParameters.isEmpty()) {
        return parameters;
      }

      LOGGER.warn("Could not find query parameters {} using configured types.", join(", ", missingParameters));
    }

    LOGGER.debug("Getting Stored Procedure parameters types using DB metadata");
    parameters = getStoredProcedureParamTypesUsingMetadataAndValidate(connection, queryTemplate);

    return parameters;
  }

  private Map<Integer, DbType> getStoredProcedureParamTypesUsingMetadataAndValidate(DbConnection connection,
                                                                                    QueryTemplate queryTemplate)
      throws SQLException {
    Map<Integer, DbType> parameters = getStoredProcedureParamTypesUsingMetadata(connection, queryTemplate);
    List<String> missingParameters = getMissingParameters(queryTemplate, parameters);

    if (!missingParameters.isEmpty()) {
      throw new SQLException(format("Could not find query parameters %s.", join(",", missingParameters)));
    }

    return parameters;
  }

  private Map<Integer, DbType> getStoredProcedureParamTypesUsingMetadata(DbConnection connection, QueryTemplate queryTemplate)
      throws SQLException {
    DatabaseMetaData dbMetaData = connection.getJdbcConnection().getMetaData();
    String storedProcedureName;

    Pair<String, Boolean> storedOperationResult = analyzeStoredOperation(queryTemplate.getSqlText());
    storedProcedureName = storedOperationResult.getFirst();
    String storedProcedureOwner = getStoreProcedureOwner(queryTemplate.getSqlText()).orElse(null);
    String storedProcedureParentOwner = getStoredProcedureParentOwner(queryTemplate.getSqlText()).orElse(null);

    if (dbMetaData.storesUpperCaseIdentifiers()) {
      storedProcedureName = storedProcedureName.toUpperCase();

      if (storedProcedureOwner != null) {
        storedProcedureOwner = storedProcedureOwner.toUpperCase();
      }

      if (storedProcedureParentOwner != null) {
        storedProcedureParentOwner = storedProcedureParentOwner.toUpperCase();
      }
    }

    try (ResultSet procedureColumns =
        connection.getProcedureColumns(storedProcedureName, storedProcedureOwner, storedProcedureParentOwner,
                                       connection.getJdbcConnection().getCatalog())) {

      return getStoredProcedureParamTypes(connection, storedProcedureName, procedureColumns, storedOperationResult.getSecond());
    }
  }

  private Map<Integer, DbType> getStoredProcedureParamTypes(DbConnection connection, String storedProcedureName,
                                                            ResultSet procedureColumns, boolean isFunction)
      throws SQLException {
    Map<Integer, DbType> paramTypes = new HashMap<>();

    int position = 1;

    while (procedureColumns.next()) {
      if (!isFunction && procedureColumns.getShort(COLUMN_TYPE_INDEX) == PROCEDURE_COLUMN_RETURN_COLUMN_TYPE) {
        continue;
      }

      int typeId = procedureColumns.getInt(TYPE_ID_COLUMN_INDEX);
      String typeName = procedureColumns.getString(TYPE_NAME_COLUMN_INDEX);
      String parameterName = procedureColumns.getString(PARAM_NAME_COLUMN_INDEX);

      LOGGER.debug("Resolved parameter type: Store procedure: {}, Name: {}, Index: {}, Type ID: {}, Type Name: {}",
                   storedProcedureName, parameterName, position, typeId, typeName);

      // TODO - MULE-15241 : Fix how DB Connector chooses ResolvedTypes
      DbType dbType;
      try {
        Optional<DbType> vendorDbType = connection.getDbTypeByVendor(typeName, procedureColumns);
        dbType = vendorDbType.orElse(dbTypeManager.lookup(connection, typeId, typeName));
      } catch (UnknownDbTypeException e) {
        // If Oracle, we give a last chance without the scheme name appended on the typeName
        dbType = silentlyCheckForOracleWithoutScheme(connection, typeId, typeName);
      }

      paramTypes.put(position, dbType);
      position++;
    }

    return paramTypes;
  }

  private List<String> getMissingParameters(QueryTemplate queryTemplate, Map<Integer, DbType> paramTypes) {
    return queryTemplate.getParams().stream()
        .filter(queryParam -> !paramTypes.containsKey(queryParam.getIndex()))
        .map(QueryParam::getName)
        .collect(Collectors.toList());
  }

  private boolean shouldForceParametersTypes() {
    return parseBoolean(getProperty(FORCE_SP_PARAM_TYPES, "false"));
  }

  private Map<Integer, DbType> getParameterTypesFromConfiguration(QueryTemplate queryTemplate,
                                                                  List<ParameterType> parameterTypesConfigured) {
    Map<Integer, DbType> paramTypes = new HashMap<>();

    for (QueryParam queryParam : queryTemplate.getParams()) {

      Optional<ParameterType> type =
          parameterTypesConfigured.stream().filter(p -> p.getKey().equals(queryParam.getName())).findAny();

      if (type.isPresent()) {
        String parameterTypeName = type.get().getDbType().getName();

        DbType dbType;

        if (parameterTypeName == null) {
          // Use unknown data type
          dbType = UnknownDbType.getInstance();
        } else {
          dbType = type.get().getDbType();
        }

        paramTypes.put(queryParam.getIndex(), dbType);
      }
    }

    return paramTypes;
  }

  private DbType silentlyCheckForOracleWithoutScheme(DbConnection connection, int typeId, String typeName) {
    try {
      DatabaseMetaData metaData = connection.getJdbcConnection().getMetaData();

      if (metaData != null && metaData.getDriverName() != null && metaData.getDriverName().toLowerCase().contains("oracle")) {
        return dbTypeManager.lookup(connection, typeId, substringAfterLast(typeName, "."));
      }
    } catch (SQLException | UnknownDbTypeException exception) {
      // Either driver does not provide metadata access or type was not found in the type manager
      return new ResolvedDbType(typeId, typeName);
    }

    // Type was not found in the type manager, but the DB knows about it
    return new ResolvedDbType(typeId, typeName);
  }

}
