/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.util;

import java.io.IOException;
import java.io.InputStream;

import org.mule.runtime.core.api.util.func.CheckedRunnable;
import org.mule.runtime.core.api.util.func.Once;

/**
 * Input stream wrapper class that runs a callback when the EOF is reached.
 */
public class InputStreamWithEOFCallbackWrapper extends InputStream {

  private final InputStream delegate;
  private final Once.RunOnce notifier;
  private final static int EOF_VALUE = -1;

  public InputStreamWithEOFCallbackWrapper(InputStream delegate, CheckedRunnable eofCallback) {
    this.delegate = delegate;
    this.notifier = Once.of(eofCallback);
  }

  @Override
  public int read() throws IOException {
    final int byteRead = delegate.read();
    if (byteRead == EOF_VALUE) {
      notifier.runOnce();
    }
    return byteRead;
  }

  @Override
  public int read(byte[] b) throws IOException {
    final int bytesRead = delegate.read(b);
    if (bytesRead == EOF_VALUE) {
      notifier.runOnce();
    }
    return bytesRead;
  }

  @Override
  public int read(byte[] b, int off, int len) throws IOException {
    final int bytesRead = delegate.read(b, off, len);
    if (bytesRead == EOF_VALUE) {
      notifier.runOnce();
    }
    return bytesRead;
  }

  @Override
  public long skip(long n) throws IOException {
    return delegate.skip(n);
  }

  @Override
  public int available() throws IOException {
    return delegate.available();
  }

  @Override
  public void close() throws IOException {
    delegate.close();
  }

  @Override
  public void mark(int readlimit) {
    delegate.mark(readlimit);
  }

  @Override
  public void reset() throws IOException {
    delegate.reset();
  }

  @Override
  public boolean markSupported() {
    return delegate.markSupported();
  }
}
