/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.db.commons.internal.domain.type;

import static java.lang.String.format;

import org.mule.db.commons.internal.domain.connection.DbConnection;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.sql.Clob;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import org.apache.commons.io.IOUtils;
import org.mule.runtime.api.metadata.TypedValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Defines a data type for {@link Clob}
 */
public class ClobResolvedDataType extends ResolvedDbType {

  private static final Logger LOGGER = LoggerFactory.getLogger(ClobResolvedDataType.class);
  private static final String JTDS_DRIVER = "jTDS";

  public ClobResolvedDataType(int id, String name) {
    super(id, name);
  }

  @Override
  public void setParameterValue(PreparedStatement statement, int index, Object value, DbConnection connection)
      throws SQLException {
    if (value != null && !(value instanceof Clob)) {
      Reader reader = getReaderFromValue(value);

      try {
        LOGGER.debug("Creating CLOB object");

        Clob clob = statement.getConnection().createClob();
        Writer writer = clob.setCharacterStream(1);
        IOUtils.copyLarge(reader, writer);

        super.setParameterValue(statement, index, clob, connection);
      } catch (Exception e) {
        // createClob method has been add to JDBC API in version 3.0. Since we have to support any driver that works
        // with JDK 1.8 we try an alternative way to set CLOB objects.
        LOGGER.debug("Error creating CLOB object. Using alternative way to set CLOB object", e);

        handlePriorVersions(statement, index, reader);
      }
    } else {
      super.setParameterValue(statement, index, value, connection);
    }
  }

  private Reader getReaderFromValue(Object value) {
    if (value instanceof String) {
      return new StringReader((String) value);
    } else if (value instanceof InputStream) {
      return new InputStreamReader((InputStream) value);
    } else if (value instanceof TypedValue<?> && ((TypedValue<?>) value).getValue() instanceof InputStream) {
      return new InputStreamReader((InputStream) ((TypedValue<?>) value).getValue());
    } else {
      throw new IllegalArgumentException(format("Cannot create a Clob from a value of type '%s'", value.getClass()));
    }
  }

  private void handlePriorVersions(PreparedStatement statement, int index, Reader reader) throws SQLException {
    // we handle the possibility of dealing with a jTDS driver
    DatabaseMetaData metaData = statement.getConnection().getMetaData();

    if (metaData != null && metaData.getDriverName() != null && metaData.getDriverName().contains(JTDS_DRIVER)) {
      try {
        statement.setString(index, IOUtils.toString(reader));
      } catch (IOException ex) {
        LOGGER.error("Exception while reading CLOB Contents (jTDS)", ex);

        throw new SQLException("Exception while reading CLOB Contents (jTDS)", ex);
      }
    } else {
      statement.setCharacterStream(index, reader);
    }
  }

}
