/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.metadata;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.metadata.api.model.MetadataFormat.XML;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_CONFIGURATION;
import static org.mule.runtime.api.metadata.resolving.FailureCode.UNKNOWN;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.query.QueryTemplate;
import org.mule.db.commons.internal.parser.SimpleQueryTemplateParser;
import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.builder.ArrayTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.BinaryType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.StringType;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.metadata.MetadataContext;
import org.mule.runtime.api.metadata.MetadataResolvingException;
import org.mule.runtime.api.util.LazyValue;
import org.slf4j.Logger;

import java.net.URL;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.Struct;
import java.sql.Types;
import java.util.HashMap;
import java.util.Map;


public abstract class BaseDbMetadataResolver {

  private static final Logger LOGGER = getLogger(BaseDbMetadataResolver.class);

  protected BaseTypeBuilder typeBuilder;
  protected ClassTypeLoader typeLoader;
  private final LazyValue<Map<Integer, MetadataType>> dbToMetaDataType = new LazyValue<>(this::initializeDbToMetaDataType);

  protected QueryTemplate parseQuery(String query) {
    return new SimpleQueryTemplateParser().parse(query);
  }

  protected PreparedStatement getStatement(MetadataContext context, QueryTemplate query)
      throws ConnectionException, MetadataResolvingException {
    DbConnection connection = context.<DbConnection>getConnection()
        .orElseThrow(() -> new MetadataResolvingException("A connection is required to resolve Metadata but none was provided",
                                                          INVALID_CONFIGURATION));
    PreparedStatement statement;
    try {
      statement = connection.getJdbcConnection().prepareStatement(query.getSqlText());
    } catch (SQLException e) {
      throw new MetadataResolvingException(e.getMessage(), UNKNOWN, e);
    }
    return statement;
  }

  protected MetadataType getDataTypeMetadataModel(int typeId, String columnClassName) {
    if (typeId == Types.JAVA_OBJECT) {
      return typeLoader.load(columnClassName).orElse(typeBuilder.anyType().build());
    } else if (typeId == Types.STRUCT) {
      try {
        if (Struct.class.isAssignableFrom(Class.forName(columnClassName))) {
          ArrayTypeBuilder arrayTypeBuilder = BaseTypeBuilder.create(JAVA).arrayType();
          arrayTypeBuilder.of().anyType();
          return arrayTypeBuilder.build();
        } else {
          return typeLoader.load(columnClassName).orElse(typeBuilder.anyType().build());
        }
      } catch (ClassNotFoundException e) {
        e.printStackTrace();
      }

    }

    return getDataTypeMetadataModel(typeId);
  }

  protected MetadataType getDataTypeMetadataModel(int columnTypeName) {
    return dbToMetaDataType.get().getOrDefault(columnTypeName, typeBuilder.anyType().build());
  }

  private Map<Integer, MetadataType> initializeDbToMetaDataType() {
    final Map<Integer, MetadataType> initialDbToMetaDataType = new HashMap<>();

    NumberType numberType = typeBuilder.numberType().build();
    StringType stringType = typeBuilder.stringType().build();
    BinaryType binaryType = typeBuilder.binaryType().build();
    AnyType anyType = typeBuilder.anyType().build();

    initialDbToMetaDataType.put(Types.BIT, typeBuilder.booleanType().build());
    initialDbToMetaDataType.put(Types.BOOLEAN, typeBuilder.booleanType().build());

    initialDbToMetaDataType.put(Types.TINYINT, numberType);
    initialDbToMetaDataType.put(Types.SMALLINT, numberType);
    initialDbToMetaDataType.put(Types.INTEGER, numberType);
    initialDbToMetaDataType.put(Types.BIGINT, numberType);
    initialDbToMetaDataType.put(Types.FLOAT, numberType);
    initialDbToMetaDataType.put(Types.REAL, numberType);
    initialDbToMetaDataType.put(Types.DOUBLE, numberType);
    initialDbToMetaDataType.put(Types.NUMERIC, numberType);
    initialDbToMetaDataType.put(Types.DECIMAL, numberType);

    initialDbToMetaDataType.put(Types.CHAR, stringType);
    initialDbToMetaDataType.put(Types.VARCHAR, stringType);
    initialDbToMetaDataType.put(Types.LONGNVARCHAR, stringType);
    initialDbToMetaDataType.put(Types.CLOB, stringType);
    initialDbToMetaDataType.put(Types.NCHAR, stringType);
    initialDbToMetaDataType.put(Types.NVARCHAR, stringType);
    initialDbToMetaDataType.put(Types.NCLOB, stringType);

    initialDbToMetaDataType.put(Types.BINARY, binaryType);
    initialDbToMetaDataType.put(Types.VARBINARY, binaryType);
    initialDbToMetaDataType.put(Types.LONGVARBINARY, binaryType);
    initialDbToMetaDataType.put(Types.BLOB, binaryType);

    initialDbToMetaDataType.put(Types.DATE, typeBuilder.dateType().build());
    initialDbToMetaDataType.put(Types.TIMESTAMP, typeBuilder.dateType().build());
    initialDbToMetaDataType.put(Types.TIME, typeBuilder.timeType().build());

    initialDbToMetaDataType.put(Types.OTHER, typeBuilder.anyType().build());
    initialDbToMetaDataType.put(Types.JAVA_OBJECT, typeBuilder.anyType().build());
    initialDbToMetaDataType.put(Types.DISTINCT, typeBuilder.anyType().build());

    initialDbToMetaDataType.put(Types.ARRAY, typeBuilder.arrayType().of(anyType).build());

    initialDbToMetaDataType.put(Types.NULL, typeBuilder.nullType().build());

    initialDbToMetaDataType.put(Types.SQLXML, BaseTypeBuilder.create(XML).objectType().build());

    initialDbToMetaDataType.put(Types.STRUCT, typeBuilder.arrayType().of(anyType).build());
    initialDbToMetaDataType.put(Types.REF, typeLoader.load(Ref.class));
    try {
      initialDbToMetaDataType.put(Types.REF_CURSOR, typeLoader.load(ResultSet.class));
    } catch (RuntimeException e) {
      // TODO (W-11494248): This change was introduce to temporarily fix the issue generated by a dependency version in Composer
      // MDS
      LOGGER.error("Could not add metadata mapping for the Ref Cursor Type (ResultSet)");
    }
    initialDbToMetaDataType.put(Types.DATALINK, typeLoader.load(URL.class));
    initialDbToMetaDataType.put(Types.ROWID, typeLoader.load(RowId.class));

    return initialDbToMetaDataType;
  }
}
