/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.db.commons.internal;

import org.mule.db.commons.internal.domain.connection.DbConnection;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Closes a {@link ResultSet} once it has been processed
 *
 * @since 1.0
 */
public class StatementStreamingResultSetCloser {

  private static final Logger LOGGER = LoggerFactory.getLogger(StatementStreamingResultSetCloser.class);

  private final DbConnection connection;
  private final Set<ResultSet> resultSets = Collections.newSetFromMap(new ConcurrentHashMap<>());

  public StatementStreamingResultSetCloser(DbConnection connection) {
    this.connection = connection;
  }

  /**
   * Closes all tracked {@link ResultSet}s
   */
  public void closeResultSets() {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Closing {} resultSets", getOpenResultSets());
    }
    try {
      for (ResultSet resultSet : resultSets) {
        close(resultSet);
      }
    } finally {
      for (int i = 0; i < resultSets.size(); i++) {
        connection.endStreaming();
      }
      resultSets.clear();
      if (!connection.isTransactionActive()) {
        connection.release();
      }
    }
  }

  private void close(ResultSet resultSet) {
    try {
      resultSet.close();
    } catch (SQLException e) {
      LOGGER.warn("Error attempting to close resultSet", e);
    }
  }

  /**
   * Adds a resultSet for tracking in order to be able to close it later
   *
   * @param resultSet resultSet to track
   */
  public void trackResultSet(ResultSet resultSet) {
    resultSets.add(resultSet);
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("ResultSet opened, now {} active", getOpenResultSets());
    }
  }

  public int getOpenResultSets() {
    return resultSets.size();
  }
}
