/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.connection.type.resolver;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.StreamSupport.stream;

import org.mule.db.commons.internal.domain.connection.DefaultDbConnection;
import org.mule.db.commons.internal.domain.type.ResolvedDbType;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Struct;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Spliterator;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Type resolver for array entities
 *
 * @since 1.5.2
 */
// TODO: Look for impact on behaviour for DefaultDbConnection, refactor to extract Oracle' specifics (it won't break
// backwards compatibility).
public class ArrayTypeResolver implements StructAndArrayTypeResolver {

  public static final String QUERY_ALL_COLL_TYPES = "SELECT * FROM SYS.ALL_COLL_TYPES WHERE TYPE_NAME = ?";
  private static final String QUERY_OWNER_CONDITION = " AND OWNER = ?";
  private static final String ELEM_TYPE_NAME = "ELEM_TYPE_NAME";
  private DefaultDbConnection connection;

  public ArrayTypeResolver(DefaultDbConnection connection) {
    this.connection = connection;
  }

  @Override
  public void resolveLobs(Object[] elements, Integer index, String dataTypeName) throws SQLException {
    for (int i = 0; i < elements.length; i++) {
      Object element = elements[i];
      if (element instanceof Struct) {
        // if it is already an Struct there is nothing to resolve
      } else if (element instanceof Collection) {
        Object[] objects = ((List<?>) element).toArray();
        connection.doResolveLobIn(objects, index, dataTypeName);
        elements[i] = objects;
      } else if (element instanceof Iterable) {
        Spliterator<?> spliterator = ((Iterable<?>) element).spliterator();
        Object[] objects = stream(spliterator, false).toArray();
        connection.doResolveLobIn(objects, index, dataTypeName);
        elements[i] = objects;
      } else if (element instanceof Object[]) {
        connection.doResolveLobIn((Object[]) element, index, dataTypeName);
      } else {
        throw new RuntimeException(String.format("Unable to process arguments of type %s", element.getClass()));
      }
    }
  }

  @Override
  public String resolveType(String typeName) throws SQLException {
    String collectionTypeName = getTypeFor(typeName);

    if (collectionTypeName != null) {
      return collectionTypeName;
    }
    return typeName;
  }

  @Override
  public String resolveType(String typeName, ConcurrentHashMap<String, String> typeCache) throws SQLException {
    if (typeCache.containsKey(typeName)) {
      return typeCache.get(typeName);
    }
    String resolvedType = resolveType(typeName);
    typeCache.put(typeName, resolvedType);
    return resolvedType;
  }

  @Override
  public void resolveLobIn(Object[] attributes, Integer index, ResolvedDbType resolvedDbType) throws SQLException {
    for (Object attribute : attributes) {
      connection.doResolveLobIn((Object[]) attribute, index, resolvedDbType.getId(), resolvedDbType.getName());
    }
  }

  private String getTypeFor(String collectionTypeName) throws SQLException {
    String dataType = null;

    Optional<String> owner = getOwnerFrom(collectionTypeName);
    String typeName = getTypeSimpleName(collectionTypeName);

    String query = QUERY_ALL_COLL_TYPES + (owner.isPresent() ? QUERY_OWNER_CONDITION : "");

    try (PreparedStatement ps = connection.prepareStatement(query)) {
      ps.setString(1, typeName);

      if (owner.isPresent()) {
        ps.setString(2, owner.get());
      }

      try (ResultSet resultSet = ps.executeQuery()) {
        while (resultSet.next()) {
          dataType = resultSet.getString(ELEM_TYPE_NAME);
        }
      }
    }
    return dataType;
  }

  public static Optional<String> getOwnerFrom(String typeName) {
    return typeName.contains(".") ? of(typeName.substring(0, typeName.indexOf('.'))) : empty();
  }

  public static String getTypeSimpleName(String typeName) {
    if (!typeName.contains(".")) {
      return typeName;
    } else {
      return typeName.substring(typeName.indexOf('.') + 1);
    }
  }
}
