/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.api.param;

import static java.util.concurrent.TimeUnit.SECONDS;
import static org.mule.runtime.extension.api.annotation.param.display.Placement.ADVANCED_TAB;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Placement;

import java.util.concurrent.TimeUnit;

/**
 * Parameters to configure queries
 *
 * @since 1.0
 */
public class QuerySettings {



  /**
   * Indicates the minimum amount of time before the JDBC driver attempts to cancel a running statement. No timeout is used by
   * default.
   */
  @Parameter
  @Optional(defaultValue = "0")
  @Placement(tab = ADVANCED_TAB)
  private int queryTimeout = 0;

  /**
   * A {@link TimeUnit} which qualifies the {@link #queryTimeout}
   */
  @Parameter
  @Optional(defaultValue = "SECONDS")
  @Placement(tab = ADVANCED_TAB)
  private TimeUnit queryTimeoutUnit = SECONDS;

  /**
   * Indicates how many rows to fetch from the database when rows are read from a resultSet. This property is required when
   * streaming is {@code true}; in that case a default value (10) is used.
   */
  @Parameter
  @Optional
  @Placement(tab = ADVANCED_TAB)
  private Integer fetchSize;

  /**
   * Sets the limit for the maximum number of rows that any ResultSet object generated by this message processor can contain for
   * the given number. If the limit is exceeded, the excess rows are silently dropped.
   */
  @Parameter
  @Optional
  @Placement(tab = ADVANCED_TAB)
  private Integer maxRows;



  /**
   * Creates a new instance of QuerySettings
   */
  public QuerySettings() {

  }

  /**
   * Creates a new instance of Query Settings with the given arguments.
   *
   * @param queryTimeout     Indicates the minimum amount of time before the JDBC driver attempts to cancel a running statement.
   *                         No timeout is used by default.
   * @param queryTimeoutUnit A {@link TimeUnit} which qualifies the {@link #queryTimeout}
   * @param fetchSize        Indicates how many rows to fetch from the database when rows are read from a resultSet.
   * @param maxRows          * Sets the limit for the maximum number of rows that any ResultSet object generated by this message
   *                         processor can contain for the given number. If the limit is exceeded, the excess rows are silently
   *                         dropped.
   */
  public QuerySettings(int queryTimeout, TimeUnit queryTimeoutUnit, Integer fetchSize, Integer maxRows) {
    this.queryTimeout = queryTimeout;
    this.queryTimeoutUnit = queryTimeoutUnit;
    this.fetchSize = fetchSize;
    this.maxRows = maxRows;
  }


  public void copyInto(QuerySettings other) {
    other.queryTimeout = queryTimeout;
    other.queryTimeoutUnit = queryTimeoutUnit;
    other.fetchSize = fetchSize;
    other.maxRows = maxRows;
  }

  public int getQueryTimeout() {
    return queryTimeout;
  }

  public TimeUnit getQueryTimeoutUnit() {
    return queryTimeoutUnit;
  }

  public Integer getFetchSize() {
    return fetchSize;
  }

  public Integer getMaxRows() {
    return maxRows;
  }

}
