/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.connection.enhydra.wrappers;

import static org.mule.db.commons.internal.domain.connection.enhydra.wrappers.EnhydraWrapperUtils.setTLSProperties;
import static org.mule.db.commons.internal.util.ClassUtils.getFieldValue;

import org.mule.runtime.api.tls.TlsContextFactory;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Properties;

import org.enhydra.jdbc.standard.StandardDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AbstractDataSourceWrapper {

  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractDataSourceWrapper.class);


  public synchronized Connection getSecureConnectionIfPossible(String user, String password,
                                                               StandardDataSource wrappedDataSource,
                                                               TlsContextFactory tlsContextFactory)
      throws SQLException {
    try {
      Connection ret = null;
      Properties prop = new Properties();

      String url = getFieldValue(wrappedDataSource, "url", true);
      Driver driver = getFieldValue(wrappedDataSource, "driver", true);
      String driverName = getFieldValue(wrappedDataSource, "driverName", true);
      boolean loadedFromCCL = getFieldValue(wrappedDataSource, "loadedFromCCL", true);

      final boolean hasUser = user != null && !user.isEmpty();
      final boolean hasPassword = password != null && !password.isEmpty();

      if (hasUser) {
        prop.put("user", user);
      }

      if (hasPassword) {
        prop.put("password", password);
      }

      // If no user/pass credentials have been provided, use ssl as authentication service in the database.
      // Check that mtls is on, and that no user/password credentials have been provided.
      if (!hasUser && !hasPassword && tlsContextFactory != null && tlsContextFactory.isKeyStoreConfigured()
          && tlsContextFactory.isTrustStoreConfigured()) {
        prop.put("oracle.net.authentication_services", "(TCPS)");
      }

      setTLSProperties(prop, tlsContextFactory);

      if (url != null) {
        if (driver == null) {
          try {
            driver = (Driver) Class.forName(driverName).newInstance();
            loadedFromCCL = false;
            wrappedDataSource.log.debug("StandardDataSource:getConnection a new driver instance is created");
          } catch (Exception e) {
            try {
              driver = (Driver) Class.forName(driverName, true, Thread.currentThread().getContextClassLoader()).newInstance();
              loadedFromCCL = true;
            } catch (Exception e2) {
              String errorMessage = "Error trying to load driver: " + driverName + " : " + e2.getMessage();

              if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("An error occurred during getConnection execution: {}", errorMessage);
              }

              throw new SQLException(errorMessage);
            }
          }
        }

        try {
          ret = loadedFromCCL ? driver.connect(url, prop) : DriverManager.getConnection(url, prop);

          int transIsolation = wrappedDataSource.getTransactionIsolation();
          if (transIsolation >= 0) {
            ret.setTransactionIsolation(transIsolation);
          }

          wrappedDataSource.log.debug("StandardDataSource:getConnection Connection from DriverManager is returned");
        } catch (SQLException e) {
          String errorMessage = "Cannot get connection for URL " + url + " : " + e.getMessage();
          LOGGER.error(errorMessage, e);

          throw new SQLException(errorMessage, e.getSQLState(), e.getErrorCode(), e.getCause());
        }
      }

      return ret;
    } catch (IllegalAccessException | NoSuchFieldException e) {
      String errorMessage = "An error occurred during getConnection execution: " + e.getMessage();
      LOGGER.error(errorMessage, e);

      throw new SQLException(errorMessage, e);
    }
  }
}
