/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.result.row;

import org.apache.commons.io.input.ReaderInputStream;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.util.IOUtils;

import java.io.IOException;
import java.nio.charset.Charset;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.SQLException;
import java.sql.SQLXML;

import static org.mule.runtime.api.metadata.MediaType.XML;
import static org.mule.runtime.api.metadata.MediaType.BINARY;
import static org.mule.runtime.api.metadata.MediaType.TEXT;

/**
 * Maps a row using returning a case insensitive map to its values. No streams will be returned as values. This means that all of
 * the mapped values will be effectively read into memory and returned in the result.
 */
public class NonStreamingInsensitiveMapRowHandler extends InsensitiveMapRowHandler {

  public NonStreamingInsensitiveMapRowHandler(DbConnection dbConnection) {
    super(dbConnection);
  }

  public NonStreamingInsensitiveMapRowHandler(DbConnection dbConnection, Charset charset) {
    super(dbConnection, charset);
  }

  @Override
  protected TypedValue<Object> handleSqlXmlType(SQLXML value) throws SQLException {
    String stringValue = value.getString();
    value.free();
    return new TypedValue<>(stringValue, DataType.builder().type(SQLXML.class).mediaType(XML).build());
  }

  @Override
  protected TypedValue<Object> handleBlobType(Blob value) throws SQLException {
    byte[] bValue = IOUtils.toByteArray(value.getBinaryStream());
    value.free();
    return new TypedValue<>(bValue,
                            DataType.builder().type(byte[].class).mediaType(BINARY).build());
  }

  @Override
  protected TypedValue<Object> handleClobType(Clob value) throws SQLException {
    ReaderInputStream inputStream = null;
    try {
      inputStream = ReaderInputStream.builder().setReader(value.getCharacterStream()).setCharset(charset).get();
    } catch (IOException e) {
      throw new MuleRuntimeException(e);
    }
    String stringValue = IOUtils.toString(inputStream);
    value.free();
    return new TypedValue<>(stringValue, DataType.builder()
        .type(String.class)
        .mediaType(TEXT)
        .charset(charset)
        .build());
  }

}
