/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.parser.statement.detector;

import org.mule.db.commons.internal.domain.query.QueryType;

import java.util.List;

import static java.util.Arrays.asList;

/**
 * Utility class useful for detecting the Statement Type (QueryType) of SQL String.
 */
public class StatementTypeDetector {

  private final List<AbstractStatementTypeDetector> detectors = asList(new SelectStatementTypeDetector(),
                                                                       new InsertStatementTypeDetector(),
                                                                       new DeleteStatementTypeDetector(),
                                                                       new UpdateStatementTypeDetector(),
                                                                       new StoredProcedureStatementTypeDetector(),
                                                                       new TruncateStatementTypeDetector(),
                                                                       new MergeStatementTypeDetector(),
                                                                       new WithStatementTypeDetector());

  /**
   * Detects the Statement Type (QueryType) given a SQL String.
   * 
   * @param sql SQL String
   * @return Detected QueryType
   */
  public QueryType detect(String sql) throws UnknownStatementTypeException {

    return detectors
        .stream()
        .filter(x -> x.isOfType(sql))
        .findFirst().map(AbstractStatementTypeDetector::getType)
        .orElseThrow(() -> new UnknownStatementTypeException("Cannot detect the statement type for query :" + sql));

  }
}

